import React, { createContext, useContext, useState, useEffect, useCallback } from 'react';
import { useLocation } from 'react-router-dom';


const PremiumContext = createContext();

export const usePremium = () => useContext(PremiumContext);

export const PremiumProvider = ({ children }) => {
  const [isPremium, setIsPremium] = useState(false);
  const [isInTrial, setIsInTrial] = useState(false);
  const [trialDaysLeft, setTrialDaysLeft] = useState(0);
  const [trialEndDate, setTrialEndDate] = useState(null);
  const [loadingPremium, setLoadingPremium] = useState(true);
  const location = useLocation();

  const fetchPremiumStatus = useCallback(async () => {
    const token = localStorage.getItem('auth_token');
    if (!token) {
      setIsPremium(false);
      setIsInTrial(false);
      setTrialEndDate(null);
      setLoadingPremium(false);
      return;
    }

    try {
      setLoadingPremium(true);
      const response = await fetch(`${import.meta.env.VITE_API_BASE_URL}/dashboard/profile`, {
        headers: { Authorization: `Bearer ${token}` },
      });

      if (response.ok) {
        const data = await response.json();
        const user = data.data;

        setIsPremium(user.premium_status === "1" || user.premium_status === 1);
        setIsInTrial(!!user.is_in_trial);
        setTrialDaysLeft(user.trial_days_left || 0);

        // Calculate trial end date if user is in trial
        if (user.is_in_trial && user.trial_days_left) {
          const endDate = new Date();
          endDate.setDate(endDate.getDate() + user.trial_days_left);
          setTrialEndDate(endDate.toLocaleDateString('en-US', { 
            month: 'short', 
            day: 'numeric', 
            year: 'numeric' 
          }));
        } else {
          setTrialEndDate(null);
        }
      } else {
        setIsPremium(false);
        setIsInTrial(false);
        setTrialEndDate(null);
      }
    } catch (err) {
      console.error('Failed to fetch premium/trial status:', err);
    } finally {
      setLoadingPremium(false);
    }
  }, []);

  useEffect(() => {
    fetchPremiumStatus();
  }, [location.pathname, fetchPremiumStatus]);

  const refreshPremiumStatus = fetchPremiumStatus;

  return (
    <PremiumContext.Provider value={{
      isPremium,
      isInTrial,
      trialDaysLeft,
      trialEndDate,
      loadingPremium,
      refreshPremiumStatus
    }}>
      {children}
    </PremiumContext.Provider>
  );
};