import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import FlashcardStudy from './FlashcardStudy';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;

const FlashcardStudySession = () => {
  const { deckType, courseId } = useParams(); // deckType: 'due'|'all'|'mistakes', courseId: string or undefined
  const navigate = useNavigate();
  const [cards, setCards] = useState([]);
  const [deckName, setDeckName] = useState('');
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  const token = localStorage.getItem('auth_token');

  useEffect(() => {
    if (!token) {
      navigate('/login');
      return;
    }

    const fetchCards = async () => {
      setLoading(true);
      setError('');

      try {
        let url = '';
        let displayName = '';

        // Handle system decks (due, all, mistakes)
        if (deckType === 'due') {
          url = `${API_BASE_URL}/flashcards/due`;
          displayName = "Today's Reviews";
        } else if (deckType === 'all') {
          url = `${API_BASE_URL}/flashcards/all`;
          displayName = 'All Cards';
        } else if (deckType === 'mistakes') {
          url = `${API_BASE_URL}/flashcards/mistakes`;
          displayName = 'Recent Mistakes';
        }
        // Handle course-specific deck
        else if (courseId) {
          url = `${API_BASE_URL}/flashcards/course/${courseId}`;
          displayName = 'Course Deck'; // Will be updated from backend if possible
        } else {
          throw new Error('Invalid deck route');
        }

        if (!url) {
          throw new Error('Could not determine API endpoint');
        }

        const response = await fetch(url, {
          headers: {
            Authorization: `Bearer ${token}`,
            Accept: 'application/json',
          },
        });

        if (!response.ok) {
          const errData = await response.json().catch(() => ({}));
          throw new Error(errData.message || 'Failed to load cards');
        }

        const data = await response.json();

        const fetchedCards = data.cards || [];
        setCards(fetchedCards);

        // Prefer backend-provided deck name (especially for courses)
        setDeckName(data.deck_name || displayName);

        if (fetchedCards.length === 0) {
          setError('No cards available in this deck.');
        }
      } catch (err) {
        console.error('Error fetching flashcards:', err);
        setError(err.message || 'Failed to load flashcards. Please try again.');
        setCards([]);
      } finally {
        setLoading(false);
      }
    };

    fetchCards();
  }, [deckType, courseId, token, navigate]);

  // Loading State
  if (loading) {
    return (
      <div className="h-screen flex items-center justify-center bg-gradient-to-br ">
        <div className="text-center">
          <div className="w-16 h-16 border-4 border-t-indigo-600 border-gray-200 rounded-full animate-spin mx-auto mb-6"></div>
          <p className="text-xl text-gray-700">Loading your flashcards...</p>
        </div>
      </div>
    );
  }

  // Error State
  if (error && cards.length === 0) {
    return (
      <div className=" flex items-center justify-center bg-gradient-to-br ">
        <div className="text-center max-w-md mt-8">
          
          <h2 className="text-2xl font-bold text-gray-900 mb-4">Oops!</h2>
          <p className="text-gray-600 mb-8">{error}</p>
          <button
            onClick={() => navigate('/dashboard/flashcards')}
            className="px-6 py-3 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700 transition"
          >
            Back to Flashcards
          </button>
        </div>
      </div>
    );
  }

  // Empty Deck State
  if (cards.length === 0) {
    return (
      <div className="h-screen flex items-center justify-center bg-gradient-to-br from-indigo-50 to-purple-50">
        <div className="text-center max-w-md">
          <div className="text-6xl mb-6">Party</div>
          <h2 className="text-2xl font-bold text-gray-900">All caught up!</h2>
          <p className="mt-4 text-gray-600">
            {deckType === 'due'
              ? "Great job! No cards due for review today."
              : deckType === 'mistakes'
              ? "No recent mistakes — you're doing great!"
              : courseId
              ? "This course deck is empty. Add questions to get started."
              : "This deck is empty."}
          </p>
          <button
            onClick={() => navigate('/dashboard/flashcards')}
            className="mt-8 px-6 py-3 bg-indigo-600 text-white rounded-lg hover:bg-indigo-700"
          >
            Back to Dashboard
          </button>
        </div>
      </div>
    );
  }

  // Normal Study Session
  return <FlashcardStudy cards={cards} deckName={deckName} />;
};

export default FlashcardStudySession;