import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { Brain, Flame, BookOpen, AlertCircle, Plus, ChevronLeft, ChevronRight } from 'lucide-react';
import DeckCard from './DeckCard';
import StatsWidget from './StatsWidget';
import RequirePremium from '../RequirePremium';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;

const FlashcardHub = () => {
  const navigate = useNavigate();

  const [stats, setStats] = useState({
    dueToday: 0,
    totalCards: 0,
    streak: 0,
    newToday: 0,
    accuracy: 0,
    mistakes: 0,
  });

  const [courses, setCourses] = useState([]);
  const [loading, setLoading] = useState(true);
  const [loadingCourses, setLoadingCourses] = useState(false);

  // Pagination state
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalCourses, setTotalCourses] = useState(0);
  const perPage = 12;

  useEffect(() => {
    fetchStats();
    fetchCourses(1); // Always load page 1 on mount
  }, []);

  const fetchStats = async () => {
    const token = localStorage.getItem('auth_token');
    if (!token) {
      navigate('/register');
      return;
    }

    try {
      const response = await fetch(`${API_BASE_URL}/flashcards/stats`, {
        headers: {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      });

      if (response.ok) {
        const data = await response.json();
        setStats({
          dueToday: data.dueToday || 0,
          totalCards: data.totalCards || 0,
          streak: data.streak || 0,
          newToday: data.newToday || 0,
          accuracy: data.accuracy || 0,
          mistakes: data.mistakes || 0,
        });
      }
    } catch (error) {
      console.error('Error fetching stats:', error);
    }
  };

  const fetchCourses = async (page = 1) => {
    const token = localStorage.getItem('auth_token');
    if (!token) return;

    setLoadingCourses(true);

    try {
      const response = await fetch(
        `${API_BASE_URL}/courses?page=${page}&per_page=${perPage}`,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
          },
        }
      );

      if (response.ok) {
        const data = await response.json();

        if (data.success && data.courses) {
          setCourses(data.courses);
          setCurrentPage(data.current_page || 1);
          setTotalPages(data.last_page || 1);
          setTotalCourses(data.total || 0);
        } else {
          setCourses([]);
          setTotalPages(1);
          setTotalCourses(0);
        }
      } else {
        setCourses([]);
      }
    } catch (error) {
      console.error('Error fetching courses:', error);
      setCourses([]);
    } finally {
      setLoading(false);
      setLoadingCourses(false);
    }
  };

  // Handle page change
  const goToPage = (page) => {
    if (page >= 1 && page <= totalPages && page !== currentPage) {
      setCurrentPage(page);
      fetchCourses(page);
      window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  // Map icons and colors
  const courseIcons = {
    anatomy: <Brain className="w-6 h-6" />,
    physiology: <Brain className="w-6 h-6" />,
    pathology: <AlertCircle className="w-6 h-6" />,
    pharmacology: <BookOpen className="w-6 h-6" />,
    biochemistry: <Brain className="w-6 h-6" />,
    biostatistics: <BookOpen className="w-6 h-6" />,
    epidemiology: <AlertCircle className="w-6 h-6" />,
    'internal medicine': <Brain className="w-6 h-6" />,
    general: <BookOpen className="w-6 h-6" />,
  };

  const courseColors = {
    anatomy: 'purple',
    physiology: 'green',
    pathology: 'orange',
    pharmacology: 'blue',
    biochemistry: 'indigo',
    biostatistics: 'pink',
    epidemiology: 'red',
    'internal medicine': 'purple',
    general: 'indigo',
  };

  // Quick access decks
  const quickDecks = [
    {
      id: 'due',
      name: "Today's Reviews",
      count: stats.dueToday,
      icon: <Brain className="w-6 h-6" />,
      color: 'indigo',
      type: 'system',
    },
    {
      id: 'all',
      name: 'All Cards',
      count: stats.totalCards,
      icon: <BookOpen className="w-6 h-6" />,
      color: 'blue',
      type: 'system',
    },
    {
      id: 'mistakes',
      name: 'Recent Mistakes',
      count: stats.mistakes,
      icon: <AlertCircle className="w-6 h-6" />,
      color: 'red',
      type: 'system',
    },
  ];

  // Course decks
  const courseDecks = courses.map((course) => {
    const subjectKey = (course.subject || 'general').toLowerCase();
    return {
      id: course.id,
      name: course.name,
      count: course.personal_questions_count || 0,
      icon: courseIcons[subjectKey] || <BookOpen className="w-6 h-6" />,
      color: courseColors[subjectKey] || 'indigo',
      type: 'course',
    };
  });

  const allDecks = [...quickDecks, ...courseDecks];

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="w-16 h-16 border-4 border-t-indigo-600 border-gray-200 rounded-full animate-spin"></div>
      </div>
    );
  }

  return (
    <RequirePremium>
    <div className="space-y-12 pb-8">
      {/* Header */}
      <div>
        <h1 className="text-4xl font-bold text-gray-900 flex items-center gap-4">
          <Brain className="w-12 h-12 text-indigo-600" />
          Flashcards
        </h1>
        <p className="text-lg text-gray-600 mt-3">
          Master medical knowledge with spaced repetition
        </p>
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <StatsWidget
          title="Cards Due Today"
          value={stats.dueToday}
          icon={<Brain className="w-8 h-8 text-indigo-600" />}
          subtitle="Ready for review"
        />
        <StatsWidget
          title="Study Streak"
          value={stats.streak}
          icon={<Flame className="w-8 h-8 text-orange-500" />}
          subtitle="Keep it going!"
          highlight
        />
        <StatsWidget
          title="New Cards Today"
          value={stats.newToday}
          icon={<Plus className="w-8 h-8 text-green-600" />}
          subtitle="To learn"
        />
        <StatsWidget
          title="Average Accuracy"
          value={`${Math.round(stats.accuracy)}%`}
          icon={<BookOpen className="w-8 h-8 text-blue-600" />}
          subtitle="Great progress"
        />
      </div>

      {/* Decks Section */}
      <div>
        <div className="flex items-center justify-between mb-6">
          <h2 className="text-2xl font-semibold text-gray-900">Your Decks</h2>
          <button
            onClick={() => navigate('/dashboard/generator')}
            className="flex items-center gap-2 px-5 py-3 bg-indigo-600 text-white rounded-xl font-medium hover:bg-indigo-700 transition"
          >
            <Plus className="w-5 h-5" />
            Upload PDF
          </button>
        </div>

        {/* Loading courses */}
        {loadingCourses ? (
          <div className="flex justify-center py-12">
            <div className="w-10 h-10 border-4 border-t-indigo-600 border-gray-200 rounded-full animate-spin"></div>
            <p>Loading....</p>
          </div>
        ) : (
          <>
            {/* Deck Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {allDecks.map((deck) => (
                <DeckCard
                  key={deck.id}
                  name={deck.name}
                  count={deck.count}
                  icon={deck.icon}
                  color={deck.color}
                  onStart={() => {
                    if (deck.type === 'system') {
                      navigate(`/dashboard/flashcards/study/${deck.id}`);
                    } else {
                      navigate(`/dashboard/flashcards/study/course/${deck.id}/flashcard`);
                    }
                  }}
                />
              ))}
            </div>

            {/* Pagination */}
            {totalPages > 1 && (
              <div className="mt-12 flex flex-col sm:flex-row items-center justify-between gap-6 bg-gray-50 rounded-2xl p-6">
                <p className="text-gray-700 font-medium">
                  Showing {(currentPage - 1) * perPage + 1}–{Math.min(currentPage * perPage, totalCourses)} of{' '}
                  <span className="font-bold">{totalCourses}</span> course decks
                </p>

                <div className="flex items-center gap-3">
                  <button
                    onClick={() => goToPage(currentPage - 1)}
                    disabled={currentPage === 1}
                    className="p-3 rounded-xl bg-white shadow-sm disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-100 transition"
                  >
                    <ChevronLeft className="w-5 h-5" />
                  </button>

                  <div className="flex gap-2">
                    {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => (
                      <button
                        key={page}
                        onClick={() => goToPage(page)}
                        className={`px-4 py-2 rounded-lg font-medium transition ${
                          page === currentPage
                            ? 'bg-indigo-600 text-white'
                            : 'bg-white hover:bg-gray-100 text-gray-700'
                        }`}
                      >
                        {page}
                      </button>
                    ))}
                  </div>

                  <button
                    onClick={() => goToPage(currentPage + 1)}
                    disabled={currentPage === totalPages}
                    className="p-3 rounded-xl bg-white shadow-sm disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-100 transition"
                  >
                    <ChevronRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
            )}

            {/* Empty State */}
            {allDecks.length === 0 && !loadingCourses && (
              <div className="text-center py-16">
                <p className="text-xl text-gray-600 mb-4">No flashcards yet</p>
                <p className="text-gray-500 mb-8">
                  Start building your knowledge base by adding questions
                </p>
                <button
                  onClick={() => navigate('/dashboard/questions')}
                  className="px-8 py-4 bg-indigo-600 text-white rounded-xl font-semibold hover:bg-indigo-700 transition"
                >
                  Add Your First Question
                </button>
              </div>
            )}
          </>
        )}
      </div>
    </div>
    </RequirePremium>
  );
};

export default FlashcardHub;