import React, { useState, useEffect } from 'react';
import {
  BookOpen, Plus, Edit2, Trash2, Check, X, Search,
  ChevronDown, ChevronRight, Layers, Settings,
  ToggleLeft, ToggleRight
} from 'lucide-react';

const BASE_URL = 'http://localhost:8000';

const ManageSubject = () => {
  const [subjects, setSubjects] = useState([]);
  const [expandedSubject, setExpandedSubject] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  
  // Modal states
  const [showSubjectModal, setShowSubjectModal] = useState(false);
  const [showSubtopicModal, setShowSubtopicModal] = useState(false);
  const [editingSubject, setEditingSubject] = useState(null);
  const [editingSubtopic, setEditingSubtopic] = useState(null);
  const [selectedSubjectForSubtopic, setSelectedSubjectForSubtopic] = useState(null);
  
  // Form states
  const [subjectForm, setSubjectForm] = useState({ name: '', description: '', auto: true });
  const [subtopicForm, setSubtopicForm] = useState({ name: '', description: '' });
  const [processing, setProcessing] = useState(false);

  const getAuthHeaders = () => ({
    'Authorization': `Bearer ${localStorage.getItem('auth_token')}`,
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  });

  // Fetch all subjects with their subtopics
  useEffect(() => {
    fetchSubjects();
  }, []);

  const fetchSubjects = async () => {
    setLoading(true);
    setError(null);
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects`, {
        headers: getAuthHeaders(),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      const data = await response.json();
      setSubjects(data.subjects || data.data || []);
    } catch (err) {
      console.error('Error fetching subjects:', err);
      setError('Failed to load subjects. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const fetchSubtopics = async (subjectId) => {
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects/${subjectId}/subtopics`, {
        headers: getAuthHeaders(),
      });

      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);

      const result = await response.json();
      const subtopics = result.subtopics || result.data || result || [];

      setSubjects(prev => prev.map(subject =>
        subject.id === subjectId
          ? {
              ...subject,
              subtopics: subtopics,
              subtopics_count: subtopics.length
            }
          : subject
      ));
    } catch (err) {
      console.error('Error fetching subtopics:', err);
      setError('Failed to load subtopics for this subject.');
    }
  };

  const handleExpandSubject = (subjectId) => {
    if (expandedSubject === subjectId) {
      setExpandedSubject(null);
    } else {
      setExpandedSubject(subjectId);
      const subject = subjects.find(s => s.id === subjectId);
      if (!subject?.subtopics) {
        fetchSubtopics(subjectId);
      }
    }
  };

  // Subject CRUD operations
  const handleCreateSubject = async () => {
    setProcessing(true);
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify(subjectForm),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      await fetchSubjects();
      setShowSubjectModal(false);
      setSubjectForm({ name: '', description: '', auto: true });
    } catch (err) {
      console.error('Error creating subject:', err);
      setError('Failed to create subject. Please try again.');
    } finally {
      setProcessing(false);
    }
  };

  const handleUpdateSubject = async () => {
    setProcessing(true);
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects/${editingSubject.id}`, {
        method: 'PUT',
        headers: getAuthHeaders(),
        body: JSON.stringify(subjectForm),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      await fetchSubjects();
      setShowSubjectModal(false);
      setEditingSubject(null);
      setSubjectForm({ name: '', description: '', auto: true });
    } catch (err) {
      console.error('Error updating subject:', err);
      setError('Failed to update subject. Please try again.');
    } finally {
      setProcessing(false);
    }
  };

  const handleDeleteSubject = async (subjectId) => {
    if (!window.confirm('Are you sure you want to delete this subject? All associated subtopics will also be deleted.')) {
      return;
    }
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects/${subjectId}`, {
        method: 'DELETE',
        headers: getAuthHeaders(),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      await fetchSubjects();
    } catch (err) {
      console.error('Error deleting subject:', err);
      setError('Failed to delete subject. Please try again.');
    }
  };

  const toggleAutoGenerate = async (subjectId, currentValue) => {
    setSubjects(prev => prev.map(subject =>
      subject.id === subjectId
        ? { ...subject, toggling: true }
        : subject
    ));

    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects/${subjectId}/toggle-auto-generate`, {
        method: 'PUT',
        headers: getAuthHeaders(),
        body: JSON.stringify({ auto: !currentValue }),
      });

      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);

      setSubjects(prev => prev.map(subject =>
        subject.id === subjectId
          ? { ...subject, auto: !currentValue, toggling: false }
          : subject
      ));
    } catch (err) {
      console.error('Error toggling auto-generate:', err);
      setError('Failed to update auto-generate setting.');

      setSubjects(prev => prev.map(subject =>
        subject.id === subjectId
          ? { ...subject, toggling: false }
          : subject
      ));
    }
  };

  // Subtopic CRUD operations
  const handleCreateSubtopic = async () => {
    setProcessing(true);
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subjects/${selectedSubjectForSubtopic}/subtopics`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify(subtopicForm),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      await fetchSubtopics(selectedSubjectForSubtopic);
      setShowSubtopicModal(false);
      setSubtopicForm({ name: '', description: '' });
      setSelectedSubjectForSubtopic(null);
    } catch (err) {
      console.error('Error creating subtopic:', err);
      setError('Failed to create subtopic. Please try again.');
    } finally {
      setProcessing(false);
    }
  };

  const handleUpdateSubtopic = async () => {
    setProcessing(true);
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subtopics/${editingSubtopic.id}`, {
        method: 'PUT',
        headers: getAuthHeaders(),
        body: JSON.stringify(subtopicForm),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      await fetchSubtopics(editingSubtopic.subject_id);
      setShowSubtopicModal(false);
      setEditingSubtopic(null);
      setSubtopicForm({ name: '', description: '' });
    } catch (err) {
      console.error('Error updating subtopic:', err);
      setError('Failed to update subtopic. Please try again.');
    } finally {
      setProcessing(false);
    }
  };

  const handleDeleteSubtopic = async (subtopicId, subjectId) => {
    if (!window.confirm('Are you sure you want to delete this subtopic?')) {
      return;
    }
    try {
      const response = await fetch(`${BASE_URL}/api/dashboard/subtopics/${subtopicId}`, {
        method: 'DELETE',
        headers: getAuthHeaders(),
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      await fetchSubtopics(subjectId);
    } catch (err) {
      console.error('Error deleting subtopic:', err);
      setError('Failed to delete subtopic. Please try again.');
    }
  };

  const openSubjectModal = (subject = null) => {
    if (subject) {
      setEditingSubject(subject);
      setSubjectForm({
        name: subject.name,
        description: subject.description || '',
        auto: subject.auto ?? true
      });
    } else {
      setEditingSubject(null);
      setSubjectForm({ name: '', description: '', auto: true });
    }
    setShowSubjectModal(true);
  };

  const openSubtopicModal = (subjectId, subtopic = null) => {
    if (subtopic) {
      setEditingSubtopic(subtopic);
      setSubtopicForm({
        name: subtopic.name,
        description: subtopic.description || ''
      });
    } else {
      setEditingSubtopic(null);
      setSubtopicForm({ name: '', description: '' });
      setSelectedSubjectForSubtopic(subjectId);
    }
    setShowSubtopicModal(true);
  };

  const filteredSubjects = subjects.filter(subject =>
    subject.name.toLowerCase().includes(searchTerm.toLowerCase())
  );

  return (
    <div className="min-h-screen bg-gradient-to-br  p-4 md:p-8">
      {loading ? (
        <div className="flex items-center justify-center min-h-[70vh]">
          <div className="flex flex-col items-center space-y-4">
            <div className="relative">
              <div className="animate-spin rounded-full h-20 w-20 border-4 border-blue-200"></div>
              <div className="animate-spin rounded-full h-20 w-20 border-t-4 border-blue-600 absolute top-0"></div>
            </div>
            <p className="text-gray-700 text-lg font-semibold">Loading subjects...</p>
          </div>
        </div>
      ) : (
        <div className="max-w-7xl mx-auto space-y-8">
          {/* Header */}
          <div className="bg-white rounded-2xl shadow-lg p-6 md:p-8 border border-gray-100">
            <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
              <div className="flex items-center gap-3 mb-4 lg:mb-0">
                <div className="p-3 bg-gradient-to-br from-blue-500 to-indigo-600 rounded-xl shadow-lg">
                  <BookOpen className="w-8 h-8 text-white" />
                </div>
                <div>
                  <h1 className="text-3xl md:text-4xl font-bold bg-gradient-to-r from-blue-600 to-indigo-600 bg-clip-text text-transparent">
                    Manage Subjects
                  </h1>
                  <p className="text-gray-600 text-sm">Create and organize subjects and subtopics</p>
                </div>
              </div>
              <button
                onClick={() => openSubjectModal()}
                className="px-6 py-3 bg-gradient-to-r from-blue-500 to-blue-600 text-white rounded-xl font-semibold hover:from-blue-600 hover:to-blue-700 transition-all shadow-lg hover:shadow-xl flex items-center gap-2"
              >
                <Plus className="w-5 h-5" />
                Add Subject
              </button>
            </div>
          </div>

          {/* Error Display */}
          {error && (
            <div className="bg-red-50 border-2 border-red-200 rounded-xl p-4 flex items-center justify-between">
              <p className="text-red-600 font-medium">{error}</p>
              <button onClick={() => setError(null)} className="text-red-400 hover:text-red-600">
                <X className="w-5 h-5" />
              </button>
            </div>
          )}

          {/* Stats */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
              <div className="flex items-center gap-4">
                <div className="p-3 bg-blue-100 rounded-xl">
                  <BookOpen className="w-6 h-6 text-blue-600" />
                </div>
                <div>
                  <p className="text-sm text-gray-600 font-semibold uppercase tracking-wide">Total Subjects</p>
                  <p className="text-3xl font-bold text-gray-900">{subjects.length}</p>
                </div>
              </div>
            </div>
            <div className="bg-white rounded-2xl shadow-lg p-6 border border-gray-100">
              <div className="flex items-center gap-4">
                <div className="p-3 bg-purple-100 rounded-xl">
                  <Layers className="w-6 h-6 text-purple-600" />
                </div>
                <div>
                  <p className="text-sm text-gray-600 font-semibold uppercase tracking-wide">Total Subtopics</p>
                  <p className="text-3xl font-bold text-gray-900">
                    {subjects.reduce((acc, subject) => acc + (subject.subtopics_count || 0), 0)}
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Search */}
          <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-100">
            <div className="relative">
              <Search className="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search subjects..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-12 pr-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all outline-none"
              />
            </div>
          </div>

          {/* Subjects List */}
          <div className="bg-white rounded-2xl shadow-lg border border-gray-100 overflow-hidden">
            {filteredSubjects.length === 0 ? (
              <div className="text-center py-16">
                <BookOpen className="w-16 h-16 text-gray-300 mx-auto mb-4" />
                <p className="text-gray-500 text-lg">No subjects found</p>
                <button
                  onClick={() => openSubjectModal()}
                  className="mt-4 px-6 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition-colors"
                >
                  Create your first subject
                </button>
              </div>
            ) : (
              <div className="divide-y divide-gray-100">
                {filteredSubjects.map((subject) => (
                  <div key={subject.id} className="p-6 hover:bg-gray-50 transition-colors">
                    <div className="flex items-start justify-between gap-4">
                      <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                          <button
                            onClick={() => handleExpandSubject(subject.id)}
                            className="p-1 hover:bg-gray-200 rounded transition-colors"
                          >
                            {expandedSubject === subject.id ? (
                              <ChevronDown className="w-5 h-5 text-gray-600" />
                            ) : (
                              <ChevronRight className="w-5 h-5 text-gray-600" />
                            )}
                          </button>
                          <h3 className="text-xl font-bold text-gray-900">{subject.name}</h3>
                          <span className="px-2 py-1 bg-blue-100 text-blue-700 text-xs font-semibold rounded-full">
                            {subject.subtopics_count || 0} subtopics
                          </span>
                          <span className="px-2 py-1 bg-indigo-100 text-indigo-700 text-xs font-semibold rounded-full">
                            {subject.questions_count || 0} questions
                          </span>
                        </div>
                        {subject.description && (
                          <p className="text-gray-600 ml-9 mb-3">{subject.description}</p>
                        )}
                        {/* Auto-generate toggle with loading state */}
                        <div className="flex items-center gap-3 ml-9">
                          <button
                            onClick={() => toggleAutoGenerate(subject.id, subject.auto)}
                            disabled={subject.toggling}
                            className={`flex items-center gap-2 px-4 py-2 rounded-lg font-medium transition-all shadow-sm disabled:opacity-70 disabled:cursor-not-allowed ${
                              subject.toggling
                                ? 'bg-yellow-100 text-yellow-700'
                                : subject.auto
                                ? 'bg-green-100 text-green-700 hover:bg-green-200'
                                : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                            }`}
                          >
                            {subject.toggling ? (
                              <>
                                <div className="animate-spin rounded-full h-4 w-4 border-2 border-yellow-600 border-t-transparent"></div>
                                <span className="text-sm">Updating...</span>
                              </>
                            ) : subject.auto ? (
                              <>
                                <ToggleRight className="w-5 h-5 text-green-600" />
                                <span className="text-sm">Auto-generate ON</span>
                              </>
                            ) : (
                              <>
                                <ToggleLeft className="w-5 h-5 text-gray-500" />
                                <span className="text-sm">Auto-generate OFF</span>
                              </>
                            )}
                          </button>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <button
                          onClick={() => openSubtopicModal(subject.id)}
                          className="p-2 bg-purple-100 text-purple-600 rounded-lg hover:bg-purple-200 transition-colors"
                          title="Add Subtopic"
                        >
                          <Plus className="w-5 h-5" />
                        </button>
                        <button
                          onClick={() => openSubjectModal(subject)}
                          className="p-2 bg-blue-100 text-blue-600 rounded-lg hover:bg-blue-200 transition-colors"
                          title="Edit Subject"
                        >
                          <Edit2 className="w-5 h-5" />
                        </button>
                        <button
                          onClick={() => handleDeleteSubject(subject.id)}
                          className="p-2 bg-red-100 text-red-600 rounded-lg hover:bg-red-200 transition-colors"
                          title="Delete Subject"
                        >
                          <Trash2 className="w-5 h-5" />
                        </button>
                      </div>
                    </div>

                    {/* Subtopics */}
                    {expandedSubject === subject.id && (
                      <div className="mt-4 ml-9 space-y-2">
                        {!subject.subtopics ? (
                          <div className="text-center py-4">
                            <div className="animate-spin rounded-full h-6 w-6 border-2 border-blue-500 border-t-transparent mx-auto"></div>
                          </div>
                        ) : subject.subtopics.length === 0 ? (
                          <div className="text-center py-6 bg-gray-50 rounded-lg">
                            <p className="text-gray-500">No subtopics yet</p>
                            <button
                              onClick={() => openSubtopicModal(subject.id)}
                              className="mt-2 text-sm text-blue-600 hover:text-blue-700 font-medium"
                            >
                              Add the first subtopic
                            </button>
                          </div>
                        ) : (
                          subject.subtopics.map((subtopic) => (
                            <div
                              key={subtopic.id}
                              className="flex items-center justify-between p-4 bg-gray-50 rounded-lg border border-gray-200 hover:border-gray-300 transition-colors"
                            >
                              <div className="flex-1">
                                <h4 className="font-semibold text-gray-900">{subtopic.name}</h4>
                                {subtopic.description && (
                                  <p className="text-sm text-gray-600 mt-1">{subtopic.description}</p>
                                )}
                              </div>
                              <div className="flex items-center gap-2">
                                <button
                                  onClick={() => openSubtopicModal(subject.id, subtopic)}
                                  className="p-2 bg-white text-blue-600 rounded-lg hover:bg-blue-50 transition-colors"
                                  title="Edit Subtopic"
                                >
                                  <Edit2 className="w-4 h-4" />
                                </button>
                                <button
                                  onClick={() => handleDeleteSubtopic(subtopic.id, subject.id)}
                                  className="p-2 bg-white text-red-600 rounded-lg hover:bg-red-50 transition-colors"
                                  title="Delete Subtopic"
                                >
                                  <Trash2 className="w-4 h-4" />
                                </button>
                              </div>
                            </div>
                          ))
                        )}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        </div>
      )}

      {/* Subject Modal */}
      {showSubjectModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-2xl shadow-2xl max-w-lg w-full p-8">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-2xl font-bold text-gray-900">
                {editingSubject ? 'Edit Subject' : 'Add New Subject'}
              </h3>
              <button
                onClick={() => {
                  setShowSubjectModal(false);
                  setEditingSubject(null);
                }}
                className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
              >
                <X className="w-5 h-5 text-gray-500" />
              </button>
            </div>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">
                  Subject Name *
                </label>
                <input
                  type="text"
                  value={subjectForm.name}
                  onChange={(e) => setSubjectForm({ ...subjectForm, name: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 outline-none"
                  placeholder="e.g., Mathematics"
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  value={subjectForm.description}
                  onChange={(e) => setSubjectForm({ ...subjectForm, description: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-blue-500 focus:border-blue-500 resize-none outline-none"
                  rows={3}
                  placeholder="Brief description of the subject"
                />
              </div>
              <div className="flex items-center justify-between p-4 bg-blue-50 rounded-xl">
                <div>
                  <p className="font-semibold text-gray-900">Auto-generate Questions</p>
                  <p className="text-sm text-gray-600">Enable AI question generation</p>
                </div>
                <button
                  onClick={() => setSubjectForm({ ...subjectForm, auto: !subjectForm.auto })}
                  className={`p-2 rounded-lg transition-colors ${
                    subjectForm.auto ? 'bg-green-500' : 'bg-gray-300'
                  }`}
                >
                  {subjectForm.auto ? (
                    <ToggleRight className="w-6 h-6 text-white" />
                  ) : (
                    <ToggleLeft className="w-6 h-6 text-gray-600" />
                  )}
                </button>
              </div>
            </div>
            <div className="flex gap-3 mt-6">
              <button
                onClick={() => {
                  setShowSubjectModal(false);
                  setEditingSubject(null);
                }}
                className="flex-1 px-6 py-3 border-2 border-gray-200 rounded-xl font-semibold text-gray-700 hover:bg-gray-50 transition-all"
                disabled={processing}
              >
                Cancel
              </button>
              <button
                onClick={editingSubject ? handleUpdateSubject : handleCreateSubject}
                disabled={processing || !subjectForm.name.trim()}
                className="flex-1 px-6 py-3 bg-gradient-to-r from-blue-500 to-blue-600 text-white rounded-xl font-semibold hover:from-blue-600 hover:to-blue-700 transition-all shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
              >
                {processing ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent"></div>
                    Processing...
                  </>
                ) : (
                  <>
                    <Check className="w-4 h-4" />
                    {editingSubject ? 'Update' : 'Create'}
                  </>
                )}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Subtopic Modal */}
      {showSubtopicModal && (
        <div className="fixed inset-0 bg-black/50 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-2xl shadow-2xl max-w-lg w-full p-8">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-2xl font-bold text-gray-900">
                {editingSubtopic ? 'Edit Subtopic' : 'Add New Subtopic'}
              </h3>
              <button
                onClick={() => {
                  setShowSubtopicModal(false);
                  setEditingSubtopic(null);
                  setSelectedSubjectForSubtopic(null);
                }}
                className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
              >
                <X className="w-5 h-5 text-gray-500" />
              </button>
            </div>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">
                  Subtopic Name *
                </label>
                <input
                  type="text"
                  value={subtopicForm.name}
                  onChange={(e) => setSubtopicForm({ ...subtopicForm, name: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-purple-500 focus:border-purple-500 outline-none transition-all"
                  placeholder="e.g., Algebra"
                />
              </div>
              <div>
                <label className="block text-sm font-semibold text-gray-700 mb-2">
                  Description
                </label>
                <textarea
                  value={subtopicForm.description}
                  onChange={(e) => setSubtopicForm({ ...subtopicForm, description: e.target.value })}
                  className="w-full px-4 py-3 border-2 border-gray-200 rounded-xl focus:ring-2 focus:ring-purple-500 focus:border-purple-500 resize-none outline-none transition-all"
                  rows={3}
                  placeholder="Brief description of the subtopic"
                />
              </div>
            </div>
            <div className="flex gap-3 mt-6">
              <button
                onClick={() => {
                  setShowSubtopicModal(false);
                  setEditingSubtopic(null);
                  setSelectedSubjectForSubtopic(null);
                }}
                className="flex-1 px-6 py-3 border-2 border-gray-200 rounded-xl font-semibold text-gray-700 hover:bg-gray-50 transition-all"
                disabled={processing}
              >
                Cancel
              </button>
              <button
                onClick={editingSubtopic ? handleUpdateSubtopic : handleCreateSubtopic}
                disabled={processing || !subtopicForm.name.trim()}
                className="flex-1 px-6 py-3 bg-gradient-to-r from-purple-500 to-purple-600 text-white rounded-xl font-semibold hover:from-purple-600 hover:to-purple-700 transition-all shadow-lg hover:shadow-xl disabled:opacity-50 disabled:cursor-not-allowed flex items-center justify-center gap-2"
              >
                {processing ? (
                  <>
                    <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent"></div>
                    Processing...
                  </>
                ) : (
                  <>
                    <Check className="w-4 h-4" />
                    {editingSubtopic ? 'Update' : 'Create'}
                  </>
                )}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ManageSubject;