import React, { useState, useEffect } from 'react';
import { Upload, Check, X, Building, ChevronDown, BookOpen, Target, Brain, Sparkles, CheckCircle, AlertCircle, Plus } from 'lucide-react';

const UploadQuestion = () => {
  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000/api';

  // State declarations
  const [questionType, setQuestionType] = useState('mcq');
  const [vivaType, setVivaType] = useState('');
  const [options, setOptions] = useState(['', '', '', '']);
  const [correctAnswer, setCorrectAnswer] = useState(''); // Now stores option text
  const [selectedSubject, setSelectedSubject] = useState('');
  const [selectedSubtopic, setSelectedSubtopic] = useState('');
  const [newQuestion, setNewQuestion] = useState({
    question: '',
    answer: '',
    difficulty: '',
    school: ''
  });
  const [toast, setToast] = useState({ show: false, message: '', type: '' });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [subjects, setSubjects] = useState([]);
  const [subtopics, setSubtopics] = useState([]);
  const [schools, setSchools] = useState([]);
  const [loading, setLoading] = useState({
    subjects: false,
    subtopics: false,
    schools: false
  });

  // Static difficulties and viva types
  const difficulties = ['Easy', 'Medium', 'Hard'];
  const vivaTypes = [
    'Clinical Scenario',
    'Theoretical Concept',
    // 'Practical Procedure',
    // 'Case Study',
    // 'Diagnostic Question',
    // 'Treatment Protocol',
    // 'Emergency Management',
    // 'Patient Communication'
  ];

  // Toast notification component
  const Toast = ({ show, message, type, onClose }) => {
    useEffect(() => {
      if (show) {
        const timer = setTimeout(() => {
          onClose();
        }, 4000);
        return () => clearTimeout(timer);
      }
    }, [show, onClose]);

    if (!show) return null;

    return (
      <div className={`fixed top-4 right-4 z-50 p-4 rounded-xl shadow-2xl backdrop-blur-sm transform transition-all duration-300 ease-in-out ${
        show ? 'translate-y-0 opacity-100' : '-translate-y-2 opacity-0'
      } ${
        type === 'success' ? 'bg-gradient-to-r from-green-500 to-emerald-500 text-white' :
        type === 'error' ? 'bg-gradient-to-r from-red-500 to-pink-500 text-white' :
        'bg-gradient-to-r from-blue-500 to-indigo-500 text-white'
      }`}>
        <div className="flex items-center space-x-3">
          {type === 'success' ? (
            <CheckCircle className="h-5 w-5 flex-shrink-0" />
          ) : (
            <AlertCircle className="h-5 w-5 flex-shrink-0" />
          )}
          <span className="font-medium">{message}</span>
          <button
            onClick={onClose}
            className="ml-2 text-white hover:text-gray-200 transition-colors"
          >
            <X className="h-4 w-4" />
          </button>
        </div>
      </div>
    );
  };

  // Toast notification handlers
  const showToast = (message, type) => {
    setToast({ show: true, message, type });
  };

  const hideToast = () => {
    setToast({ show: false, message: '', type: '' });
  };

  // Fetch subjects
  useEffect(() => {
    const fetchSubjects = async () => {
      setLoading(prev => ({ ...prev, subjects: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubjects(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subjects:', err);
        setSubjects([]);
        showToast('Failed to load subjects', 'error');
      } finally {
        setLoading(prev => ({ ...prev, subjects: false }));
      }
    };
    fetchSubjects();
  }, [API_BASE_URL]);

  // Fetch schools
  useEffect(() => {
    const fetchSchools = async () => {
      setLoading(prev => ({ ...prev, schools: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/institutions`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSchools(data.institutions || data || []);
      } catch (err) {
        console.error('Error fetching schools:', err);
        setSchools([]);
        showToast('Failed to load institutions', 'error');
      } finally {
        setLoading(prev => ({ ...prev, schools: false }));
      }
    };
    fetchSchools();
  }, [API_BASE_URL]);

  // Fetch subtopics when subject changes
  useEffect(() => {
    const fetchSubtopics = async () => {
      if (!selectedSubject) {
        setSubtopics([]);
        setSelectedSubtopic('');
        return;
      }

      setLoading(prev => ({ ...prev, subtopics: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects/${selectedSubject}/subtopics`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubtopics(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subtopics:', err);
        setSubtopics([]);
        showToast('Failed to load subtopics', 'error');
      } finally {
        setLoading(prev => ({ ...prev, subtopics: false }));
      }
    };
    fetchSubtopics();
  }, [API_BASE_URL, selectedSubject]);

  // Handle option management
  const addOption = () => {
    if (options.length < 6) {
      setOptions([...options, '']);
    } else {
      showToast('Maximum 6 options allowed', 'error');
    }
  };

  const removeOption = (index) => {
    if (options.length > 2) {
      const newOptions = options.filter((_, i) => i !== index);
      setOptions(newOptions);
      if (correctAnswer === options[index]) {
        setCorrectAnswer('');
      }
    } else {
      showToast('Minimum 2 options required', 'error');
    }
  };

  const updateOption = (index, value) => {
    const newOptions = [...options];
    newOptions[index] = value;
    setOptions(newOptions);
    if (correctAnswer === options[index] && !value.trim()) {
      setCorrectAnswer('');
    }
  };

  // Handle subject change
  const handleSubjectChange = (e) => {
    setSelectedSubject(e.target.value);
    setSelectedSubtopic('');
  };

  // Form validation
  const validateForm = () => {
    if (!newQuestion.question.trim()) {
      showToast('Question is required', 'error');
      return false;
    }
    
    if (!selectedSubject) {
      showToast('Subject is required', 'error');
      return false;
    }
    
    if (!selectedSubtopic) {
      showToast('Subtopic is required', 'error');
      return false;
    }
    
    if (!newQuestion.difficulty) {
      showToast('Difficulty is required', 'error');
      return false;
    }

    if (questionType === 'mcq') {
      const validOptions = options.filter(option => option.trim() !== '');
      if (validOptions.length < 2) {
        showToast('At least 2 valid options are required for MCQ', 'error');
        return false;
      }
      if (!correctAnswer) {
        showToast('Correct answer is required for MCQ', 'error');
        return false;
      }
      if (!validOptions.includes(correctAnswer)) {
        showToast('Selected correct answer is not a valid option', 'error');
        return false;
      }
    } else if (questionType === 'viva') {
      if (!vivaType) {
        showToast('Viva type is required', 'error');
        return false;
      }
      if (!newQuestion.answer.trim()) {
        showToast('Answer/Explanation is required for Viva questions', 'error');
        return false;
      }
    }

    return true;
  };

  // Handle form submission
  const handleUploadQuestion = async () => {
    if (!validateForm()) return;
    
    setIsSubmitting(true);
    
    try {
      const questionData = {
        userData: localStorage.getItem('user_data'),
        question_type: questionType,
        question: newQuestion.question,
        subject_id: selectedSubject,
        subtopic_id: selectedSubtopic,
        difficulty: newQuestion.difficulty,
        institution: newQuestion.school
      };

      if (questionType === 'mcq') {
        questionData.options = options.filter(option => option.trim() !== '');
        questionData.correct_answer = correctAnswer; // Send option text
      } else {
        questionData.viva_type = vivaType;
        questionData.answer = newQuestion.answer;
      }

      const response = await fetch(`${API_BASE_URL}/dashboard/questions`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify(questionData)
      });

      if (!response.ok) {
        throw new Error(`HTTP error! Status: ${response.status}`);
      }

      await response.json();
      
      // Reset form
      setQuestionType('mcq');
      setVivaType('');
      setOptions(['', '', '', '']);
      setCorrectAnswer('');
      setSelectedSubject('');
      setSelectedSubtopic('');
      setNewQuestion({
        question: '',
        answer: '',
        difficulty: '',
        school: ''
      });

      showToast('Question uploaded successfully!', 'success');
    } catch (error) {
      console.error('Error uploading question:', error);
      showToast('Failed to upload question. Please try again.', 'error');
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="min-h-screen p-4 sm:p-6 lg:p-8">
      <Toast show={toast.show} message={toast.message} type={toast.type} onClose={hideToast} />
      
      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <div className="text-center mb-8">
          <div className="inline-flex items-center justify-center w-16 h-16 bg-gradient-to-br from-white/20 to-white/10 backdrop-blur-sm rounded-2xl mb-4 shadow-lg border border-white/20">
            <Upload className="h-8 w-8 text-blue-600" />
          </div>
          <h1 className="text-4xl font-bold text-gray-800 mb-2">Upload Question</h1>
          <p className="text-gray-600 text-lg">Create engaging questions for your students</p>
        </div>

        {/* Main Form */}
        <div className="bg-white/95 backdrop-blur-sm rounded-3xl shadow-2xl p-8 border border-gray-200">
          <div className="space-y-8">
            {/* Question Type Selection */}
            <div className="space-y-4">
              <label className="block text-lg font-semibold text-gray-800">
                <Brain className="inline h-5 w-5 mr-2 text-blue-600" />
                Question Type <span className="text-red-500">*</span>
              </label>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div 
                  className={`relative p-6 rounded-2xl border-2 cursor-pointer transition-all duration-300 ${
                    questionType === 'mcq' 
                      ? 'border-blue-500 bg-blue-50 shadow-lg scale-105' 
                      : 'border-gray-200 hover:border-gray-300 hover:shadow-md'
                  }`}
                  onClick={() => setQuestionType('mcq')}
                >
                  <input
                    type="radio"
                    name="question_type"
                    value="mcq"
                    checked={questionType === 'mcq'}
                    onChange={() => setQuestionType('mcq')}
                    className="absolute top-4 right-4 w-4 h-4 text-blue-600"
                    required
                  />
                  <div className="flex items-center space-x-4">
                    <div className="p-3 bg-gradient-to-br from-blue-500 to-cyan-500 rounded-xl">
                      <Target className="h-6 w-6 text-white" />
                    </div>
                    <div>
                      <h3 className="font-semibold text-gray-800 text-lg">Multiple Choice</h3>
                      <p className="text-gray-600 text-sm">Create MCQ with multiple options</p>
                    </div>
                  </div>
                </div>
                
                <div 
                  className={`relative p-6 rounded-2xl border-2 cursor-pointer transition-all duration-300 ${
                    questionType === 'viva' 
                      ? 'border-purple-500 bg-purple-50 shadow-lg scale-105' 
                      : 'border-gray-200 hover:border-gray-300 hover:shadow-md'
                  }`}
                  onClick={() => setQuestionType('viva')}
                >
                  <input
                    type="radio"
                    name="question_type"
                    value="viva"
                    checked={questionType === 'viva'}
                    onChange={() => setQuestionType('viva')}
                    className="absolute top-4 right-4 w-4 h-4 text-purple-600"
                    required
                  />
                  <div className="flex items-center space-x-4">
                    <div className="p-3 bg-gradient-to-br from-purple-500 to-pink-500 rounded-xl">
                      <BookOpen className="h-6 w-6 text-white" />
                    </div>
                    <div>
                      <h3 className="font-semibold text-gray-800 text-lg">Viva Question</h3>
                      <p className="text-gray-600 text-sm">Create open-ended questions</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Viva Type Selection */}
            {questionType === 'viva' && (
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Sparkles className="inline h-5 w-5 mr-2 text-purple-600" />
                  Viva Type <span className="text-red-500">*</span>
                </label>
                <div className="relative">
                  <select
                    value={vivaType}
                    onChange={(e) => setVivaType(e.target.value)}
                    className="w-full p-4 pr-10 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-purple-500 focus:ring-4 focus:ring-purple-100 transition-all duration-300 appearance-none"
                    required
                  >
                    <option value="">Select Viva Type</option>
                    {vivaTypes.map(type => (
                      <option key={type} value={type}>{type}</option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                </div>
              </div>
            )}

            {/* Question Input */}
            <div className="space-y-4">
              <label className="block text-lg font-semibold text-gray-800">
                <BookOpen className="inline h-5 w-5 mr-2 text-green-600" />
                Question <span className="text-red-500">*</span>
              </label>
              <textarea
                value={newQuestion.question}
                onChange={(e) => setNewQuestion({...newQuestion, question: e.target.value})}
                className="w-full p-4 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-green-500 focus:ring-4 focus:ring-green-100 transition-all duration-300 resize-none"
                rows="4"
                placeholder="Enter your question here..."
                required
              />
            </div>

            {/* MCQ Options */}
            {questionType === 'mcq' && (
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Target className="inline h-5 w-5 mr-2 text-blue-600" />
                  Options <span className="text-red-500">*</span>
                </label>
                <div className="space-y-3">
                  {options.map((option, index) => (
                    <div key={index} className="flex items-center space-x-3 group">
                      <div className="flex-shrink-0 w-8 h-8 bg-gradient-to-br from-blue-500 to-purple-600 rounded-full flex items-center justify-center text-white font-semibold text-sm">
                        {String.fromCharCode(65 + index)}
                      </div>
                      <input
                        type="text"
                        value={option}
                        onChange={(e) => updateOption(index, e.target.value)}
                        className="flex-1 p-3 text-gray-700 bg-white border-2 border-gray-200 rounded-xl focus:border-blue-500 focus:ring-4 focus:ring-blue-100 transition-all duration-300"
                        placeholder={`Option ${String.fromCharCode(65 + index)}`}
                        required={index < 2}
                      />
                      {options.length > 2 && (
                        <button
                          type="button"
                          onClick={() => removeOption(index)}
                          className="p-2 text-red-400 hover:text-red-600 hover:bg-red-50 rounded-lg transition-all duration-200 opacity-0 group-hover:opacity-100"
                        >
                          <X className="h-5 w-5" />
                        </button>
                      )}
                    </div>
                  ))}
                  <button
                    type="button"
                    onClick={addOption}
                    className="flex items-center space-x-2 text-blue-600 hover:text-blue-700 font-medium transition-colors duration-200"
                    disabled={options.length >= 6}
                  >
                    <Plus className="h-5 w-5" />
                    <span>Add Option</span>
                  </button>
                </div>
              </div>
            )}

            {/* Correct Answer for MCQ */}
            {questionType === 'mcq' && (
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Check className="inline h-5 w-5 mr-2 text-green-600" />
                  Correct Answer <span className="text-red-500">*</span>
                </label>
                <div className="relative">
                  <select
                    value={correctAnswer}
                    onChange={(e) => setCorrectAnswer(e.target.value)}
                    className="w-full p-4 pr-10 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-green-500 focus:ring-4 focus:ring-green-100 transition-all duration-300 appearance-none"
                    required
                  >
                    <option value="">Select Correct Answer</option>
                    {options.map((option, index) => (
                      option.trim() && (
                        <option key={index} value={option}>
                          {String.fromCharCode(65 + index)}. {option}
                        </option>
                      )
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                </div>
              </div>
            )}

            {/* Answer/Explanation for Viva */}
            {questionType === 'viva' && (
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <BookOpen className="inline h-5 w-5 mr-2 text-purple-600" />
                  Answer/Explanation <span className="text-red-500">*</span>
                </label>
                <textarea
                  value={newQuestion.answer}
                  onChange={(e) => setNewQuestion({...newQuestion, answer: e.target.value})}
                  className="w-full p-4 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-purple-500 focus:ring-4 focus:ring-purple-100 transition-all duration-300 resize-none"
                  rows="4"
                  placeholder="Enter the answer or explanation here..."
                  required
                />
              </div>
            )}

            {/* Subject, Subtopic, Difficulty, and School */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <BookOpen className="inline h-5 w-5 mr-2 text-indigo-600" />
                  Subject <span className="text-red-500">*</span>
                </label>
                <div className="relative">
                  <select
                    value={selectedSubject}
                    onChange={handleSubjectChange}
                    className="w-full p-4 pr-10 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-indigo-500 focus:ring-4 focus:ring-indigo-100 transition-all duration-300 appearance-none"
                    disabled={loading.subjects}
                    required
                  >
                    <option value="">
                      {loading.subjects ? 'Loading...' : 'Select Subject'}
                    </option>
                    {subjects.map(subject => (
                      <option key={subject.id} value={subject.id}>
                        {subject.name}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Target className="inline h-5 w-5 mr-2 text-cyan-600" />
                  Subtopic <span className="text-red-500">*</span>
                </label>
                <div className="relative">
                  <select
                    value={selectedSubtopic}
                    onChange={(e) => setSelectedSubtopic(e.target.value)}
                    className="w-full p-4 pr-10 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-cyan-500 focus:ring-4 focus:ring-cyan-100 transition-all duration-300 appearance-none"
                    disabled={!selectedSubject || loading.subtopics}
                    required
                  >
                    <option value="">
                      {loading.subtopics ? 'Loading...' : 
                       !selectedSubject ? 'Select subject first' : 'Select Subtopic'}
                    </option>
                    {subtopics.map(subtopic => (
                      <option key={subtopic.id} value={subtopic.id}>
                        {subtopic.name}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Brain className="inline h-5 w-5 mr-2 text-pink-600" />
                  Difficulty <span className="text-red-500">*</span>
                </label>
                <div className="relative">
                  <select
                    value={newQuestion.difficulty}
                    onChange={(e) => setNewQuestion({...newQuestion, difficulty: e.target.value})}
                    className="w-full p-4 pr-10 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-pink-500 focus:ring-4 focus:ring-pink-100 transition-all duration-300 appearance-none"
                    required
                  >
                    <option value="">Select Difficulty</option>
                    {difficulties.map(difficulty => (
                      <option key={difficulty} value={difficulty}>
                        {difficulty}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                </div>
              </div>

              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Building className="inline h-5 w-5 mr-2 text-orange-600" />
                  Institution
                </label>
                <div className="relative">
                  <select
                    value={newQuestion.school}
                    onChange={(e) => setNewQuestion({...newQuestion, school: e.target.value})}
                    className="w-full p-4 pr-10 text-gray-700 bg-white border-2 border-gray-200 rounded-2xl focus:border-orange-500 focus:ring-4 focus:ring-orange-100 transition-all duration-300 appearance-none"
                    disabled={loading.schools}
                  >
                    <option value="">
                      {loading.schools ? 'Loading...' : 'Select Institution (Optional)'}
                    </option>
                    {schools.map(school => (
                      <option key={school.id} value={school.id}>
                        {school.name}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-5 w-5 text-gray-400" />
                </div>
              </div>
            </div>

            {/* Upload Button */}
            <div className="flex justify-center pt-8">
              <button
                onClick={handleUploadQuestion}
                disabled={isSubmitting}
                className={`group relative px-8 py-4 bg-gradient-to-br from-blue-600 via-blue-700 to-green-600 text-white font-semibold rounded-2xl shadow-lg hover:shadow-xl transform transition-all duration-300 ${
                  isSubmitting 
                    ? 'opacity-75 cursor-not-allowed' 
                    : 'hover:scale-105 active:scale-95'
                }`}
              >
                <div className="flex items-center space-x-3">
                  {isSubmitting ? (
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                  ) : (
                    <Upload className="h-5 w-5 group-hover:rotate-12 transition-transform duration-300" />
                  )}
                  <span className="text-lg">
                    {isSubmitting ? 'Uploading...' : 'Upload Question'}
                  </span>
                </div>
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default UploadQuestion;