import React, { useState, useEffect } from 'react';
import { Award, CheckCircle, XCircle, Clock, RefreshCw, Play } from 'lucide-react';
import axios from 'axios';
import RequirePremium from './RequirePremium';

const Quiz = () => {
  // State management
  const [subjects, setSubjects] = useState([]);
  const [subtopics, setSubtopics] = useState([]);
  const [currentQuiz, setCurrentQuiz] = useState(null);
  const [currentQuestionIndex, setCurrentQuestionIndex] = useState(0);
  const [userAnswers, setUserAnswers] = useState([]);
  const [quizCompleted, setQuizCompleted] = useState(false);
  const [quizResults, setQuizResults] = useState(null);
  const [timeLeft, setTimeLeft] = useState(null);
  const [quizStarted, setQuizStarted] = useState(false);
  const [quizSessionId, setQuizSessionId] = useState(null);
  const [toast, setToast] = useState({ show: false, message: '', type: 'info' });
  const [currentAnswerFeedback, setCurrentAnswerFeedback] = useState(null); // For Learning Mode
  const [isAnswerSubmitted, setIsAnswerSubmitted] = useState(false); // Track answer submission in Learning Mode

  // Loading states
  const [loading, setLoading] = useState({
    subjects: false,
    subtopics: false,
    generating: false,
    submitting: false,
    checkingAnswer: false,
  });

  // Quiz settings with mode
  const [quizSettings, setQuizSettings] = useState({
    subject: 'all',
    subtopic: 'all',
    difficulty: 'all',
    questionCount: 10,
    timeLimit: 30, // minutes, only for Quiz Mode
    mode: 'quiz', // 'quiz' or 'learning'
  });

  // API configuration
  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000/api';
  const difficulties = ['Easy', 'Medium', 'Hard'];

  // Toast notification helper
  const showToast = (message, type = 'info') => {
    setToast({ show: true, message, type });
    setTimeout(() => setToast({ show: false, message: '', type: 'info' }), 3000);
  };

  // Get auth headers
  const getAuthHeaders = () => ({
    Authorization: `Bearer ${localStorage.getItem('auth_token')}`,
    'Content-Type': 'application/json',
    Accept: 'application/json',
  });

  // Fetch subjects on component mount
  useEffect(() => {
    const fetchSubjects = async () => {
      setLoading((prev) => ({ ...prev, subjects: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects`, {
          headers: getAuthHeaders(),
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubjects(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subjects:', err);
        setSubjects([]);
        showToast('Failed to load subjects', 'error');
      } finally {
        setLoading((prev) => ({ ...prev, subjects: false }));
      }
    };
    fetchSubjects();
  }, [API_BASE_URL]);

  // Fetch subtopics when subject changes
  useEffect(() => {
    const fetchSubtopics = async () => {
      if (!quizSettings.subject || quizSettings.subject === 'all') {
        setSubtopics([]);
        setQuizSettings((prev) => ({ ...prev, subtopic: 'all' }));
        return;
      }

      setLoading((prev) => ({ ...prev, subtopics: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects/${quizSettings.subject}/subtopics`, {
          headers: getAuthHeaders(),
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubtopics(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subtopics:', err);
        setSubtopics([]);
        showToast('Failed to load subtopics', 'error');
      } finally {
        setLoading((prev) => ({ ...prev, subtopics: false }));
      }
    };
    fetchSubtopics();
  }, [API_BASE_URL, quizSettings.subject]);

  // Timer effect for Quiz Mode
  useEffect(() => {
    let interval;
    if (quizStarted && timeLeft > 0 && !quizCompleted && quizSettings.mode === 'quiz') {
      interval = setInterval(() => {
        setTimeLeft((prev) => {
          if (prev <= 1) {
            submitQuiz(true); // Auto-submit when time runs out
            return 0;
          }
          return prev - 1;
        });
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [quizStarted, timeLeft, quizCompleted, quizSettings.mode]);

  // Generate quiz
  const generateQuiz = async () => {
    setLoading((prev) => ({ ...prev, generating: true }));
  
    try {
      const payload = {
        subject: quizSettings.subject !== 'all' ? quizSettings.subject : null,
        subtopic: quizSettings.subtopic !== 'all' ? quizSettings.subtopic : null,
        difficulty: quizSettings.difficulty !== 'all' ? quizSettings.difficulty : null,
        question_count: quizSettings.questionCount,
      };
  
      if (quizSettings.mode === 'quiz') {
        payload.time_limit = quizSettings.timeLimit;
      }
  
      const response = await fetch(`${API_BASE_URL}/quiz/start`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify(payload),
      });
  
      // Always try to parse JSON, even on error responses
      let data;
      try {
        data = await response.json();
      } catch (parseError) {
        // If JSON parsing fails (rare, but possible on malformed response)
        throw new Error('Invalid response from server');
      }
  
      // Handle non-2xx responses (your Laravel API returns 400, 404, 422, etc.)
      if (!response.ok) {
        // Extract the message from API response
        const errorMessage =
          data.message ||
          data.error ||
          data.errors?.[Object.keys(data.errors || {})[0]]?.[0] || // for validation errors
          `HTTP error! Status: ${response.status}`;
  
        showToast(errorMessage, 'error');
        return;
      }
  
      // Success case
      if (!data.session_id || !data.questions || data.questions.length === 0) {
        showToast('No questions available for the selected criteria', 'error');
        return;
      }
  
      const questionsWithoutAnswers = data.questions.map((q) => ({
        id: q.id,
        question: q.question,
        options: q.options,
        subject: q.subject,
        subtopic: q.subtopic,
        difficulty: q.difficulty,
      }));
  
      setQuizSessionId(data.session_id);
      setCurrentQuiz(questionsWithoutAnswers);
      setCurrentQuestionIndex(0);
      setUserAnswers(new Array(questionsWithoutAnswers.length).fill(''));
      setQuizCompleted(false);
      setQuizResults(null);
      setTimeLeft(quizSettings.mode === 'quiz' ? quizSettings.timeLimit * 60 : null);
      setQuizStarted(true);
      setCurrentAnswerFeedback(null);
      setIsAnswerSubmitted(false);
  
      showToast(
        `Started ${quizSettings.mode === 'quiz' ? 'Quiz' : 'Learning'} Mode with ${questionsWithoutAnswers.length} questions!`,
        'success'
      );
    } catch (err) {
      console.error('Error generating quiz:', err);
  
      // This catch is for network errors, timeouts, etc.
      const networkErrorMsg = err.message.includes('Failed to fetch')
        ? 'Network error: Please check your internet connection'
        : 'Failed to connect to server';
  
      showToast(err.message || networkErrorMsg, 'error');
    } finally {
      setLoading((prev) => ({ ...prev, generating: false }));
    }
  };

  // Check answer in Learning Mode
  const checkAnswer = async (selectedOption) => {
    if (!quizSessionId || !currentQuiz) return;

    setLoading((prev) => ({ ...prev, checkingAnswer: true }));
    try {
      const response = await fetch(`${API_BASE_URL}/quiz/check-answer`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify({
          session_id: quizSessionId,
          question_id: currentQuiz[currentQuestionIndex].id,
          selected_answer: selectedOption,
        }),
      });

      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);

      const data = await response.json();
      setCurrentAnswerFeedback({
        isCorrect: data.is_correct,
        correctAnswer: data.correct_answer,
        explanation: data.explanation,
      });
      setIsAnswerSubmitted(true);

      showToast(data.is_correct ? 'Correct!' : 'Incorrect. See explanation below.', data.is_correct ? 'success' : 'error');
    } catch (err) {
      console.error('Error checking answer:', err);
      showToast('Failed to check answer', 'error');
    } finally {
      setLoading((prev) => ({ ...prev, checkingAnswer: false }));
    }
  };

  // Handle answer selection
  const handleAnswerSelect = (selectedOption) => {
    if (quizSettings.mode === 'learning' && !isAnswerSubmitted) {
      checkAnswer(selectedOption);
      const newAnswers = [...userAnswers];
      newAnswers[currentQuestionIndex] = selectedOption;
      setUserAnswers(newAnswers);
    } else if (quizSettings.mode === 'quiz') {
      const newAnswers = [...userAnswers];
      newAnswers[currentQuestionIndex] = selectedOption;
      setUserAnswers(newAnswers);
    }
  };

  // Next question
  const nextQuestion = () => {
    if (quizSettings.mode === 'learning') {
      if (!isAnswerSubmitted) {
        showToast('Please select an answer first', 'info');
        return;
      }
      if (currentQuestionIndex < currentQuiz.length - 1) {
        setCurrentQuestionIndex((prev) => prev + 1);
        setIsAnswerSubmitted(false);
        setCurrentAnswerFeedback(null);
      } else {
        setQuizCompleted(true);
        setQuizStarted(false);
        showToast('Learning session completed!', 'success');
      }
    } else {
      if (currentQuestionIndex < currentQuiz.length - 1) {
        setCurrentQuestionIndex((prev) => prev + 1);
      } else {
        submitQuiz();
      }
    }
  };

  // Previous question
  const prevQuestion = () => {
    if (currentQuestionIndex > 0) {
      setCurrentQuestionIndex((prev) => prev - 1);
      setIsAnswerSubmitted(false);
      setCurrentAnswerFeedback(null);
    }
  };

  // Submit quiz for Quiz Mode
  const submitQuiz = async (timeExpired = false) => {
    if (!quizSessionId || !currentQuiz) return;

    setLoading((prev) => ({ ...prev, submitting: true }));

    try {
      const timeSpentSeconds = (quizSettings.timeLimit * 60) - timeLeft;
      const answersPayload = currentQuiz.map((question, index) => ({
        question_id: question.id,
        selected_answer: userAnswers[index] || null,
      }));

      const response = await fetch(`${API_BASE_URL}/quiz/submit`, {
        method: 'POST',
        headers: getAuthHeaders(),
        body: JSON.stringify({
          session_id: quizSessionId,
          answers: answersPayload,
          time_expired: timeExpired,
          time_spent: timeSpentSeconds,
        }),
      });

      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);

      const data = await response.json();

      if (data.status !== 'success') {
        throw new Error(data.message || 'Failed to submit quiz');
      }

      const results = data.results;

      setQuizResults({
        score: results.score,
        correctAnswers: results.correct_answers,
        totalQuestions: results.total_questions,
        timeSpent: results.time_spent,
        detailedResults: results.detailed_results,
        passed: results.score >= 70,
      });

      setQuizCompleted(true);
      setQuizStarted(false);

      showToast(`Quiz completed! Score: ${results.score}%`, results.score >= 70 ? 'success' : 'info');
    } catch (err) {
      console.error('Error submitting quiz:', err);
      showToast(err.message || 'Failed to submit quiz', 'error');
    } finally {
      setLoading((prev) => ({ ...prev, submitting: false }));
    }
  };

  // Reset quiz
  const resetQuiz = () => {
    setCurrentQuiz(null);
    setCurrentQuestionIndex(0);
    setUserAnswers([]);
    setQuizCompleted(false);
    setQuizResults(null);
    setTimeLeft(null);
    setQuizStarted(false);
    setQuizSessionId(null);
    setCurrentAnswerFeedback(null);
    setIsAnswerSubmitted(false);
  };

  // Format time
  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  // Results view (Quiz Mode only) - FINAL VERSION WITH ALWAYS-VISIBLE EXPLANATION
  if (quizCompleted && quizResults && quizSettings.mode === 'quiz') {
    return (
      <div className="space-y-6">
        <h1 className="text-3xl font-bold text-gray-800">Quiz Results</h1>
        <div className="bg-white p-6 rounded-lg shadow-md">
          <div className="text-center mb-6">
            <div className={`w-24 h-24 mx-auto mb-4 rounded-full flex items-center justify-center ${quizResults.passed ? 'bg-green-100' : 'bg-red-100'}`}>
              <Award className={`${quizResults.passed ? 'text-green-600' : 'text-red-600'}`} size={48} />
            </div>
            <h2 className="text-2xl font-bold text-gray-800 mb-2">Quiz Completed!</h2>
            <p className="text-gray-600">
              You scored {quizResults.score}% ({quizResults.correctAnswers}/{quizResults.totalQuestions})
            </p>
            <p className="text-sm text-gray-500 mt-1">
              Time spent: {Math.floor(quizResults.timeSpent / 60)}:{(quizResults.timeSpent % 60).toString().padStart(2, '0')}
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-8">
            <div className="bg-blue-50 p-4 rounded-lg text-center">
              <p className="text-sm text-blue-600">Total Questions</p>
              <p className="text-2xl font-bold text-blue-800">{quizResults.totalQuestions}</p>
            </div>
            <div className="bg-green-50 p-4 rounded-lg text-center">
              <p className="text-sm text-green-600">Correct</p>
              <p className="text-2xl font-bold text-green-800">{quizResults.correctAnswers}</p>
            </div>
            <div className="bg-red-50 p-4 rounded-lg text-center">
              <p className="text-sm text-red-600">Incorrect</p>
              <p className="text-2xl font-bold text-red-800">{quizResults.totalQuestions - quizResults.correctAnswers}</p>
            </div>
            <div className="bg-purple-50 p-4 rounded-lg text-center">
              <p className="text-sm text-purple-600">Score</p>
              <p className="text-2xl font-bold text-purple-800">{quizResults.score}%</p>
            </div>
          </div>

          {/* Detailed Review - Explanation ALWAYS shown when available */}
          {quizResults.detailedResults && quizResults.detailedResults.length > 0 && (
            <div className="space-y-6 border-t pt-6">
              <h3 className="text-xl font-semibold text-gray-800">Answer Review</h3>
              {quizResults.detailedResults.map((result, index) => {
                const userAnswer = result.user_answer;
                const correctAnswer = result.correct_answer;
                const isCorrect = result.is_correct;

                return (
                  <div key={result.question_id} className="border border-gray-200 rounded-xl p-6 bg-gray-50">
                    <div className="flex items-start justify-between mb-4">
                      <span className="font-medium text-gray-700">Question {index + 1}</span>
                      <span className={`px-3 py-1 rounded-full text-sm font-medium ${isCorrect ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>
                        {isCorrect ? 'Correct' : 'Incorrect'}
                      </span>
                    </div>

                    <p className="text-gray-800 font-medium mb-4">{result.question_text}</p>

                    {/* Options with visual feedback */}
                    <div className="space-y-3 mb-4">
                      {result.options.map((option, optIndex) => {
                        const letter = String.fromCharCode(65 + optIndex);
                        const isUserAnswer = option === userAnswer;
                        const isCorrectAnswer = option === correctAnswer;

                        return (
                          <div
                            key={optIndex}
                            className={`p-3 rounded-lg border-2 flex items-center justify-between transition-all ${
                              isCorrectAnswer
                                ? 'bg-green-50 border-green-400'
                                : isUserAnswer && !isCorrect
                                ? 'bg-red-50 border-red-400'
                                : 'bg-white border-gray-200'
                            }`}
                          >
                            <div className="flex items-center space-x-3">
                              <span className="font-medium text-gray-700">{letter}.</span>
                              <span className="text-gray-800">{option}</span>
                            </div>
                            <div className="flex items-center space-x-2">
                              {isCorrectAnswer && <CheckCircle className="text-green-600" size={20} />}
                              {isUserAnswer && !isCorrect && <XCircle className="text-red-600" size={20} />}
                            </div>
                          </div>
                        );
                      })}
                    </div>

                    {/* Always show Correct Answer + Explanation section */}
                    <div className="mt-4 space-y-3">
                      {/* Highlight correct answer for all questions */}
                      <div className="bg-emerald-50 border border-emerald-300 rounded-lg p-4">
                        <p className="text-sm font-semibold text-emerald-900">
                          Correct Answer: <span className="font-bold text-emerald-800">{correctAnswer}</span>
                        </p>
                      </div>

                      {/* Always show explanation if it exists */}
                      {result.explanation ? (
                        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                          <p className="text-sm font-semibold text-blue-900 mb-1">Explanation</p>
                          <p className="text-blue-800 text-sm leading-relaxed">{result.explanation}</p>
                        </div>
                      ) : (
                        <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
                          <p className="text-sm text-gray-600 italic">No explanation provided for this question.</p>
                        </div>
                      )}
                    </div>
                  </div>
                );
              })}
            </div>
          )}

          <div className="flex justify-center mt-8">
            <button
              onClick={resetQuiz}
              className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-8 py-3 rounded-xl font-semibold hover:from-indigo-700 hover:to-purple-700 transition-all duration-300 flex items-center space-x-2 shadow-lg"
            >
              <RefreshCw size={20} />
              <span>Take Another Quiz</span>
            </button>
          </div>
        </div>
      </div>
    );
  }

  // Quiz or Learning Mode in progress
  if (currentQuiz && !quizCompleted) {
    const currentQuestion = currentQuiz[currentQuestionIndex];
    const progress = ((currentQuestionIndex + 1) / currentQuiz.length) * 100;

    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <h1 className="text-3xl font-bold text-gray-800">{quizSettings.mode === 'quiz' ? 'Quiz' : 'Learning'} Mode</h1>
          <div className="flex items-center space-x-4">
            {quizSettings.mode === 'quiz' && (
              <div className="flex items-center space-x-2 text-red-600">
                <Clock size={20} />
                <span className="font-bold">{formatTime(timeLeft)}</span>
              </div>
            )}
            <button
              onClick={resetQuiz}
              className="text-red-500 hover:text-red-700 flex items-center space-x-2"
            >
              <XCircle size={20} />
              <span>Exit {quizSettings.mode === 'quiz' ? 'Quiz' : 'Session'}</span>
            </button>
          </div>
        </div>

        <div className="bg-white p-6 rounded-lg shadow-md">
          <div className="mb-6">
            <div className="flex justify-between items-center mb-2">
              <span className="text-sm text-gray-600">Question {currentQuestionIndex + 1} of {currentQuiz.length}</span>
              <span className="text-sm text-gray-600">{Math.round(progress)}% Complete</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div className="bg-blue-600 h-2 rounded-full transition-all duration-300" style={{ width: `${progress}%` }}></div>
            </div>
          </div>

          <div className="mb-6">
            <div className="flex flex-wrap gap-2 mb-4">
              <span className={`px-2 py-1 rounded-full text-xs font-medium ${currentQuestion.difficulty === 'Easy' ? 'bg-green-100 text-green-800' : currentQuestion.difficulty === 'Medium' ? 'bg-yellow-100 text-yellow-800' : 'bg-red-100 text-red-800'}`}>
                {currentQuestion.difficulty}
              </span>
              <span className="px-2 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-medium">
                {currentQuestion.subject}
              </span>
              {currentQuestion.subtopic && (
                <span className="px-2 py-1 bg-purple-100 text-purple-800 rounded-full text-xs font-medium">
                  {currentQuestion.subtopic}
                </span>
              )}
            </div>

            <h3 className="text-lg font-semibold mb-4">Question:</h3>
            <p className="text-gray-700 mb-6">{currentQuestion.question}</p>

            <div className="space-y-2 mb-6">
              {currentQuestion.options.map((option, index) => (
                <button
                  key={index}
                  onClick={() => handleAnswerSelect(option)}
                  disabled={quizSettings.mode === 'learning' && isAnswerSubmitted}
                  className={`w-full p-3 text-left rounded-lg border transition-all ${
                    quizSettings.mode === 'learning' && isAnswerSubmitted
                      ? option === currentAnswerFeedback?.correctAnswer
                        ? 'bg-green-50 border-green-200'
                        : option === userAnswers[currentQuestionIndex] && !currentAnswerFeedback?.isCorrect
                        ? 'bg-red-50 border-red-200'
                        : 'bg-gray-50 border-gray-200'
                      : userAnswers[currentQuestionIndex] === option
                      ? 'bg-blue-50 border-blue-300 text-blue-800'
                      : 'bg-gray-50 border-gray-200 hover:bg-gray-100'
                  }`}
                >
                  <div className="flex items-center space-x-3">
                    <span className="flex-shrink-0 w-6 h-6 rounded-full border border-gray-300 flex items-center justify-center text-sm font-medium">
                      {String.fromCharCode(65 + index)}
                    </span>
                    <span>{option}</span>
                    {quizSettings.mode === 'learning' && isAnswerSubmitted && option === currentAnswerFeedback?.correctAnswer && (
                      <CheckCircle className="text-green-600" size={16} />
                    )}
                    {quizSettings.mode === 'learning' && isAnswerSubmitted && option === userAnswers[currentQuestionIndex] && !currentAnswerFeedback?.isCorrect && (
                      <XCircle className="text-red-600" size={16} />
                    )}
                  </div>
                </button>
              ))}
            </div>

            {quizSettings.mode === 'learning' && isAnswerSubmitted && currentAnswerFeedback && (
              <div className="mb-6">
                <div className={`p-3 rounded-lg ${currentAnswerFeedback.isCorrect ? 'bg-green-50' : 'bg-red-50'}`}>
                  <p className="text-sm font-medium">
                    {currentAnswerFeedback.isCorrect ? 'Correct!' : 'Incorrect!'}
                  </p>
                  {!currentAnswerFeedback.isCorrect && (
                    <>
                      <p className="text-sm mt-2">
                        Correct Answer: {currentAnswerFeedback.correctAnswer}
                      </p>
                      {currentAnswerFeedback.explanation && (
                        <p className="text-sm mt-2">Explanation: {currentAnswerFeedback.explanation}</p>
                      )}
                    </>
                  )}
                </div>
              </div>
            )}
          </div>

          <div className="flex justify-between">
            <button
              onClick={prevQuestion}
              disabled={currentQuestionIndex === 0}
              className="bg-gray-500 text-white px-4 py-2 rounded-md hover:bg-gray-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
            >
              Previous
            </button>

            <button
              onClick={nextQuestion}
              disabled={loading.checkingAnswer || (quizSettings.mode === 'quiz' && loading.submitting)}
              className="bg-blue-500 text-white px-6 py-2 rounded-md hover:bg-blue-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center space-x-2"
            >
              {(loading.checkingAnswer || loading.submitting) && (
                <RefreshCw className="animate-spin" size={16} />
              )}
              <span>
                {quizSettings.mode === 'learning'
                  ? currentQuestionIndex === currentQuiz.length - 1
                    ? 'Finish Session'
                    : 'Next Question'
                  : currentQuestionIndex === currentQuiz.length - 1
                  ? 'Submit Quiz'
                  : 'Next Question'}
              </span>
            </button>
          </div>
        </div>
      </div>
    );
  }

  // Quiz setup
  return (
    <RequirePremium>
    <div className="space-y-6">
      <h1 className="text-3xl font-bold text-gray-800">Create Quiz</h1>
      <div className="bg-white p-6 rounded-lg shadow-md">
        <div className="mb-6">
          <h2 className="text-xl font-semibold mb-4">Quiz Settings</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Mode</label>
              <select
                value={quizSettings.mode}
                onChange={(e) => setQuizSettings({ ...quizSettings, mode: e.target.value })}
                className="w-full p-2 border border-gray-300 rounded-md"
              >
                <option value="quiz">Quiz Mode</option>
                <option value="learning">Learning Mode</option>
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Subject</label>
              <select
                value={quizSettings.subject}
                onChange={(e) => setQuizSettings({ ...quizSettings, subject: e.target.value })}
                className="w-full p-2 border border-gray-300 rounded-md"
                disabled={loading.subjects}
              >
                <option value="all">All Subjects</option>
                {subjects.map((subject) => (
                  <option key={subject.id || subject} value={subject.id || subject}>
                    {subject.name || subject}
                  </option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Subtopic</label>
              <select
                value={quizSettings.subtopic}
                onChange={(e) => setQuizSettings({ ...quizSettings, subtopic: e.target.value })}
                className="w-full p-2 border border-gray-300 rounded-md"
                disabled={loading.subtopics || quizSettings.subject === 'all'}
              >
                <option value="all">All Subtopics</option>
                {subtopics.map((subtopic) => (
                  <option key={subtopic.id || subtopic} value={subtopic.id || subtopic}>
                    {subtopic.name || subtopic}
                  </option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Difficulty</label>
              <select
                value={quizSettings.difficulty}
                onChange={(e) => setQuizSettings({ ...quizSettings, difficulty: e.target.value })}
                className="w-full p-2 border border-gray-300 rounded-md"
              >
                <option value="all">All Difficulties</option>
                {difficulties.map((difficulty) => (
                  <option key={difficulty} value={difficulty}>{difficulty}</option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Number of Questions</label>
              <select
                value={quizSettings.questionCount}
                onChange={(e) => setQuizSettings({ ...quizSettings, questionCount: parseInt(e.target.value) })}
                className="w-full p-2 border border-gray-300 rounded-md"
              >
                <option value={5}>5 Questions</option>
                <option value={10}>10 Questions</option>
                <option value={15}>15 Questions</option>
                <option value={30}>30 Questions</option>
                <option value={50}>50 Questions</option>
                
              </select>
            </div>

            {quizSettings.mode === 'quiz' && (
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Time Limit (minutes)</label>
                <select
                  value={quizSettings.timeLimit}
                  onChange={(e) => setQuizSettings({ ...quizSettings, timeLimit: parseInt(e.target.value) })}
                  className="w-full p-2 border border-gray-300 rounded-md"
                >
                  <option value={15}>15 Minutes</option>
                  <option value={30}>30 Minutes</option>
                  <option value={45}>45 Minutes</option>
                  <option value={60}>1 Hour</option>
                  <option value={90}>1.5 Hours</option>
                </select>
              </div>
            )}
          </div>
        </div>

        <button
          onClick={generateQuiz}
          disabled={loading.generating}
          className="bg-blue-500 text-white px-6 py-3 rounded-md hover:bg-blue-600 transition-colors flex items-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          {loading.generating ? (
            <>
              <RefreshCw className="animate-spin" size={20} />
              <span>Generating...</span>
            </>
          ) : (
            <>
              <Play size={20} />
              <span>Start {quizSettings.mode === 'quiz' ? 'Quiz' : 'Learning Session'}</span>
            </>
          )}
        </button>

        <div className="mt-6">
          <h3 className="text-lg font-semibold mb-4">Available Questions</h3>
          {loading.subjects ? (
            <div className="text-center py-4">Loading subjects...</div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {subjects.map((subject) => (
                <div key={subject.id || subject} className="bg-gray-50 p-4 rounded-lg">
                  <h4 className="font-medium text-gray-800">{subject.name || subject}</h4>
                  <p className="text-sm text-gray-600">{subject.questions_count || '0'} questions available</p>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>

      {/* Toast notification */}
      {toast.show && (
        <div className={`fixed top-4 right-4 px-4 py-2 rounded-md text-white z-50 ${toast.type === 'success' ? 'bg-green-500' : toast.type === 'error' ? 'bg-red-500' : 'bg-blue-500'}`}>
          {toast.message}
        </div>
      )}
    </div>
    </RequirePremium>
  );
};

export default Quiz;