import React, { useState, useEffect } from 'react';
import { BookOpen, Search, Filter, Download, ArrowLeft, CheckCircle, Calendar, FileText, Eye, EyeOff } from 'lucide-react';

const QuestionsLibrary = ({ onNavigateBack }) => {
  const [questions, setQuestions] = useState([]);
  const [filteredQuestions, setFilteredQuestions] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCourse, setSelectedCourse] = useState('');
  const [courses, setCourses] = useState([]);
  const [showAnswers, setShowAnswers] = useState({});
  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;
  // Fetch questions from the API
  const fetchQuestions = async () => {
    setLoading(true);
    try {
      const response = await fetch(`${API_BASEURL}/api/user-questions`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token')}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`Failed to fetch questions: ${response.status}`);
      }

      const result = await response.json();
      const questionsData = result.questions || [];
      
      setQuestions(questionsData);
      setFilteredQuestions(questionsData);
      
      // Extract unique courses
      const uniqueCourses = [...new Set(questionsData.map(q => q.course_name).filter(Boolean))];
      setCourses(uniqueCourses);
      
    } catch (error) {
      console.error('Error fetching questions:', error);
      setError('Failed to load questions. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchQuestions();
  }, []);

  // Filter questions based on search and course selection
  useEffect(() => {
    let filtered = questions;

    if (searchTerm) {
      filtered = filtered.filter(q => 
        q.question.toLowerCase().includes(searchTerm.toLowerCase()) ||
        (q.course_name && q.course_name.toLowerCase().includes(searchTerm.toLowerCase()))
      );
    }

    if (selectedCourse) {
      filtered = filtered.filter(q => q.course_name === selectedCourse);
    }

    setFilteredQuestions(filtered);
  }, [searchTerm, selectedCourse, questions]);

  const toggleShowAnswer = (questionId) => {
    setShowAnswers(prev => ({
      ...prev,
      [questionId]: !prev[questionId]
    }));
  };

  const exportQuestions = (courseQuestions, courseName) => {
    const dataToExport = courseQuestions.map(q => ({
      question: q.question,
      options: JSON.parse(q.options || '[]'),
      correct_answer: q.answer,
      explanation: q.explanation,
      course_name: q.course_name,
      created_at: q.created_at
    }));

    const json = JSON.stringify(dataToExport, null, 2);
    const blob = new Blob([json], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${courseName || 'questions'}-export.json`;
    a.click();
    URL.revokeObjectURL(url);
  };

  const groupQuestionsByCourse = () => {
    const grouped = {};
    filteredQuestions.forEach(question => {
      const courseName = question.course_name || 'Uncategorized';
      if (!grouped[courseName]) {
        grouped[courseName] = [];
      }
      grouped[courseName].push(question);
    });
    return grouped;
  };

  const groupedQuestions = groupQuestionsByCourse();

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-50 via-white to-purple-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-indigo-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading your questions...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-indigo-50 via-white to-purple-50 font-sans text-gray-800">
      <div className="max-w-7xl mx-auto p-6 lg:p-12">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <div className="flex items-center space-x-4">
            
            <div>
              <h1 className="text-3xl sm:text-4xl font-extrabold bg-gradient-to-r from-indigo-600 to-purple-600 bg-clip-text text-transparent">
                My Questions Library
              </h1>
              <p className="text-gray-600 mt-2">
                Manage and review your generated questions
              </p>
            </div>
          </div>
        </div>

        {/* Search and Filter Controls */}
        <div className="bg-white/80 backdrop-blur-sm rounded-2xl shadow-lg border border-white/20 p-6 mb-8">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
              <input
                type="text"
                placeholder="Search questions or courses..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-xl focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 outline-none transition-all"
              />
            </div>
            <div className="relative">
              <Filter className="w-5 h-5 text-gray-400 absolute left-3 top-1/2 transform -translate-y-1/2" />
              <select
                value={selectedCourse}
                onChange={(e) => setSelectedCourse(e.target.value)}
                className="pl-10 pr-8 py-3 border border-gray-300 rounded-xl focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 outline-none transition-all bg-white"
              >
                <option value="">All Courses</option>
                {courses.map(course => (
                  <option key={course} value={course}>{course}</option>
                ))}
              </select>
            </div>
          </div>
        </div>

        {/* Error Display */}
        {error && (
          <div className="bg-red-50 border border-red-200 rounded-2xl p-4 flex items-center space-x-3 mb-8">
            <div className="w-2 h-2 rounded-full bg-red-500 flex-shrink-0" />
            <span className="text-red-700">{error}</span>
          </div>
        )}

        {/* Questions Display */}
        {Object.keys(groupedQuestions).length === 0 ? (
          <div className="bg-white/80 backdrop-blur-sm rounded-3xl shadow-xl border border-white/20 p-12 text-center">
            <BookOpen className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-gray-700 mb-2">No Questions Found</h3>
            <p className="text-gray-500 mb-6">
              {searchTerm || selectedCourse 
                ? "No questions match your current search or filter criteria."
                : "You haven't generated any questions yet. Start by uploading a document!"
              }
            </p>
            {!searchTerm && !selectedCourse && (
              <button
                onClick={onNavigateBack}
                className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white px-6 py-3 rounded-xl font-semibold hover:from-indigo-700 hover:to-purple-700 transition-all duration-300"
              >
                Generate Questions
              </button>
            )}
          </div>
        ) : (
          <div className="space-y-8">
            {Object.entries(groupedQuestions).map(([courseName, courseQuestions]) => (
              <div key={courseName} className="bg-white/80 backdrop-blur-sm rounded-3xl shadow-xl border border-white/20 overflow-hidden">
                {/* Course Header */}
                <div className="bg-gradient-to-r from-indigo-500 to-purple-600 text-white p-6 flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <BookOpen className="w-6 h-6" />
                    <div>
                      <h3 className="text-xl font-bold">{courseName}</h3>
                      <p className="text-indigo-100">{courseQuestions.length} questions</p>
                    </div>
                  </div>
                  <button
                    onClick={() => exportQuestions(courseQuestions, courseName)}
                    className="flex items-center space-x-2 bg-white/20 hover:bg-white/30 px-4 py-2 rounded-xl transition-all duration-300"
                  >
                    <Download className="w-4 h-4" />
                    <span>Export</span>
                  </button>
                </div>

                {/* Questions List */}
                <div className="p-6 space-y-6">
                  {courseQuestions.map((question, index) => {
                    const options = JSON.parse(question.options || '[]');
                    const isAnswerVisible = showAnswers[question.id];

                    return (
                      <div key={question.id} className="bg-gradient-to-r from-gray-50 to-gray-100/50 rounded-2xl p-6 border border-gray-200/50">
                        <div className="flex items-start justify-between mb-4">
                          <h4 className="font-bold text-lg text-gray-800 leading-relaxed flex-1 mr-4">
                            {index + 1}. {question.question}
                          </h4>
                          <div className="flex items-center space-x-2">
                            {question.created_at && (
                              <div className="flex items-center text-xs text-gray-500">
                                <Calendar className="w-3 h-3 mr-1" />
                                {new Date(question.created_at).toLocaleDateString()}
                              </div>
                            )}
                            <button
                              onClick={() => toggleShowAnswer(question.id)}
                              className={`flex items-center space-x-1 px-3 py-1 rounded-lg text-sm font-medium transition-all duration-300 ${
                                isAnswerVisible
                                  ? 'bg-green-100 text-green-700 hover:bg-green-200'
                                  : 'bg-gray-200 text-gray-700 hover:bg-gray-300'
                              }`}
                            >
                              {isAnswerVisible ? <EyeOff className="w-4 h-4" /> : <Eye className="w-4 h-4" />}
                              <span>{isAnswerVisible ? 'Hide' : 'Show'} Answer</span>
                            </button>
                          </div>
                        </div>

                        <div className="grid gap-3 mb-4">
                          {options.map((option, optIndex) => (
                            <div
                              key={optIndex}
                              className={`p-4 rounded-xl transition-all duration-200 ${
                                isAnswerVisible && optIndex === question.answer
                                  ? 'bg-gradient-to-r from-green-100 to-emerald-100 border-2 border-green-300 shadow-sm'
                                  : 'bg-white border border-gray-200'
                              }`}
                            >
                              <div className="flex items-start space-x-3">
                                <span
                                  className={`font-bold text-sm px-2 py-1 rounded-lg ${
                                    isAnswerVisible && optIndex === question.answer
                                      ? 'bg-green-500 text-white'
                                      : 'bg-gray-200 text-gray-700'
                                  }`}
                                >
                                  {String.fromCharCode(65 + optIndex)}
                                </span>
                                <span className="flex-1">{option}</span>
                                {isAnswerVisible && optIndex === question.answer && (
                                  <CheckCircle className="w-5 h-5 text-green-600 flex-shrink-0" />
                                )}
                              </div>
                            </div>
                          ))}
                        </div>

                        {question.explanation && isAnswerVisible && (
                          <div className="bg-blue-50 border border-blue-200 rounded-xl p-4">
                            <div className="flex items-start space-x-2">
                              <div className="w-2 h-2 rounded-full bg-blue-500 mt-2 flex-shrink-0" />
                              <div>
                                <strong className="text-blue-800">Explanation:</strong>
                                <p className="text-blue-700 mt-1">{question.explanation}</p>
                              </div>
                            </div>
                          </div>
                        )}
                      </div>
                    );
                  })}
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  );
};

export default QuestionsLibrary;