import React, { useState, useEffect, useMemo } from 'react';
import { 
  BookOpen, Users, Clock, Search, ChevronRight, Loader, AlertCircle,
  Award, ArrowRight, ArrowLeft, RotateCcw, Play, Pause, CheckCircle, XCircle, X,
  ChevronLeft, ChevronRight as ChevronRightIcon
} from 'lucide-react';
import { useDebouncedCallback } from 'use-debounce';
import RequirePremium from './RequirePremium';

const PublicQuiz = () => {
  const [courses, setCourses] = useState([]);
  const [subjects, setSubjects] = useState([]);
  const [subtopics, setSubtopics] = useState([]);
  const [selectedSubject, setSelectedSubject] = useState("all");
  const [selectedSubtopic, setSelectedSubtopic] = useState("all");
  const [searchQuery, setSearchQuery] = useState("");
  const [loading, setLoading] = useState({
    courses: true,
    subjects: true,
    subtopics: false,
  });
  const [error, setError] = useState("");

  // Pagination state
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalCourses, setTotalCourses] = useState(0);
  const perPage = 12; // Matches backend default

  // Quiz/Learning states
  const [selectedCourse, setSelectedCourse] = useState(null);
  const [showModeModal, setShowModeModal] = useState(false);
  const [showQuizConfig, setShowQuizConfig] = useState(false);
  const [currentMode, setCurrentMode] = useState(null);
  const [questions, setQuestions] = useState([]);
  const [currentQuestionIndex, setCurrentQuestionIndex] = useState(0);
  const [userAnswers, setUserAnswers] = useState({});
  const [showResults, setShowResults] = useState(false);
  const [isReviewing, setIsReviewing] = useState(false);
  const [showExplanation, setShowExplanation] = useState(false);
  const [quizDuration, setQuizDuration] = useState(15);
  const [timeLeft, setTimeLeft] = useState(0);
  const [isQuizActive, setIsQuizActive] = useState(false);

  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000/api';

  const getAuthHeaders = () => ({
    'Authorization': `Bearer ${localStorage.getItem('auth_token') || ''}`,
    'Accept': 'application/json',
  });

  const debouncedSetSearchQuery = useDebouncedCallback(
    (value) => setSearchQuery(value),
    500
  );

  // Reset to page 1 when filters change
  useEffect(() => {
    setCurrentPage(1);
  }, [selectedSubject, selectedSubtopic, searchQuery]);

  // Fetch subjects
  useEffect(() => {
    const fetchSubjects = async () => {
      setLoading(prev => ({ ...prev, subjects: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects`, {
          headers: getAuthHeaders(),
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubjects(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subjects:', err);
        setSubjects([]);
        setError('Failed to load subjects');
      } finally {
        setLoading(prev => ({ ...prev, subjects: false }));
      }
    };
    fetchSubjects();
  }, [API_BASE_URL]);

  // Fetch subtopics
  useEffect(() => {
    const fetchSubtopics = async () => {
      if (!selectedSubject || selectedSubject === 'all') {
        setSubtopics([]);
        setSelectedSubtopic('all');
        return;
      }

      setLoading(prev => ({ ...prev, subtopics: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects/${selectedSubject}/subtopics`, {
          headers: getAuthHeaders(),
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubtopics(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subtopics:', err);
        setSubtopics([]);
      } finally {
        setLoading(prev => ({ ...prev, subtopics: false }));
      }
    };
    fetchSubtopics();
  }, [API_BASE_URL, selectedSubject]);

  // Fetch courses with filters + pagination
  useEffect(() => {
    const fetchCourses = async () => {
      setLoading(prev => ({ ...prev, courses: true }));
      setError("");

      try {
        let url = `${API_BASE_URL}/public-quiz`;
        const params = new URLSearchParams();

        params.append("page", currentPage);
        params.append("per_page", perPage);

        if (selectedSubject && selectedSubject !== "all") {
          params.append("subject_id", selectedSubject);
        }
        if (selectedSubtopic && selectedSubtopic !== "all") {
          params.append("subtopic_id", selectedSubtopic);
        }
        if (searchQuery.trim()) {
          params.append("search", searchQuery.trim());
        }

        url += `?${params.toString()}`;

        const response = await fetch(url, {
          headers: getAuthHeaders(),
        });

        if (!response.ok) {
          const errData = await response.json().catch(() => ({}));
          throw new Error(errData.message || `Failed to fetch courses (${response.status})`);
        }

        const data = await response.json();

        // Handle Laravel pagination response
        if (data.courses && data.current_page !== undefined) {
          setCourses(data.courses);
          setCurrentPage(data.current_page || 1);
          setTotalPages(data.last_page || 1);
          setTotalCourses(data.total || 0);
        } else {
          // Fallback for non-paginated or different structure
          const courseList = data.courses || data.data || (Array.isArray(data) ? data : []);
          setCourses(courseList);
          setTotalPages(1);
          setTotalCourses(courseList.length);
        }
      } catch (err) {
        console.error("Error fetching courses:", err);
        setError(err.message || "Failed to load courses");
        setCourses([]);
        setTotalPages(1);
        setTotalCourses(0);
      } finally {
        setLoading(prev => ({ ...prev, courses: false }));
      }
    };

    fetchCourses();
  }, [API_BASE_URL, selectedSubject, selectedSubtopic, searchQuery, currentPage]);

  // Timer logic
  useEffect(() => {
    if (!isQuizActive || timeLeft <= 0) return;

    const interval = setInterval(() => {
      setTimeLeft((prev) => {
        if (prev <= 1) {
          clearInterval(interval);
          setIsQuizActive(false);
          setShowResults(true);
          return 0;
        }
        return prev - 1;
      });
    }, 1000);

    return () => clearInterval(interval);
  }, [isQuizActive, timeLeft]);

  const fetchQuestions = async (courseId) => {
    setError('');
    try {
      const response = await fetch(`${API_BASE_URL}/personal_questions?course_id=${courseId}`, {
        headers: getAuthHeaders(),
      });
      if (!response.ok) throw new Error('Failed to load questions');
      const data = await response.json();
      const questionList = Array.isArray(data) ? data : [];
      setQuestions(questionList);
      return questionList;
    } catch (err) {
      setError('Failed to load questions: ' + err.message);
      setQuestions([]);
      return [];
    }
  };

  const handleCourseClick = (course) => {
    setSelectedCourse(course);
    setShowModeModal(true);
  };

  const handleModeSelect = async (mode) => {
    setCurrentMode(mode);
    setShowModeModal(false);

    const data = await fetchQuestions(selectedCourse.id);
    if (data.length === 0) {
      setError('No questions available for this course');
      resetAll();
      return;
    }

    if (mode === 'quiz') {
      setShowQuizConfig(true);
    } else {
      setCurrentQuestionIndex(0);
      setUserAnswers({});
      setShowExplanation(false);
    }
  };

  const startQuizMode = () => {
    const totalSeconds = quizDuration * 60;
    setTimeLeft(totalSeconds);
    setIsQuizActive(true);
    setCurrentQuestionIndex(0);
    setUserAnswers({});
    setShowResults(false);
    setIsReviewing(false);
    setShowQuizConfig(false);
  };

  const handleAnswerSelect = (questionId, selectedAnswer) => {
    setUserAnswers(prev => ({ ...prev, [questionId]: selectedAnswer }));
    if (currentMode === 'learning') setShowExplanation(true);
  };

  const nextQuestion = () => {
    if (currentQuestionIndex < questions.length - 1) {
      setCurrentQuestionIndex(prev => prev + 1);
      setShowExplanation(false);
    } else if (currentMode === 'quiz' && !isReviewing) {
      setIsQuizActive(false);
      setShowResults(true);
    }
  };

  const previousQuestion = () => {
    if (currentQuestionIndex > 0) {
      setCurrentQuestionIndex(prev => prev - 1);
      if (currentMode === 'learning') {
        setShowExplanation(!!userAnswers[questions[currentQuestionIndex - 1]?.id]);
      }
    }
  };

  const calculateResults = useMemo(() => {
    if (questions.length === 0) return null;
    const total = questions.length;
    const answered = Object.keys(userAnswers).length;
    const correct = questions.filter(q => userAnswers[q.id] === q.answer).length;
    const percentage = Math.round((correct / total) * 100);
    return { total, answered, correct, wrong: answered - correct, unanswered: total - answered, percentage };
  }, [questions, userAnswers]);

  const resetAll = () => {
    setSelectedCourse(null);
    setCurrentMode(null);
    setQuestions([]);
    setCurrentQuestionIndex(0);
    setUserAnswers({});
    setShowResults(false);
    setIsReviewing(false);
    setShowQuizConfig(false);
    setShowModeModal(false);
    setTimeLeft(0);
    setIsQuizActive(false);
    setShowExplanation(false);
    setError('');
  };

  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  const currentQuestion = questions[currentQuestionIndex];

  // Pagination helpers
  const goToPage = (page) => {
    if (page >= 1 && page <= totalPages && page !== currentPage) {
      setCurrentPage(page);
    }
  };

  const renderPageNumbers = () => {
    const pages = [];
    const maxVisible = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisible / 2));
    let endPage = Math.min(totalPages, startPage + maxVisible - 1);

    if (endPage - startPage + 1 < maxVisible) {
      startPage = Math.max(1, endPage - maxVisible + 1);
    }

    for (let i = startPage; i <= endPage; i++) {
      pages.push(
        <button
          key={i}
          onClick={() => goToPage(i)}
          className={`px-4 py-2 rounded-lg font-medium transition-all ${
            i === currentPage
              ? 'bg-indigo-600 text-white shadow-md'
              : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
          }`}
        >
          {i}
        </button>
      );
    }
    return pages;
  };

  // === QUIZ CONFIG MODAL ===
  if (showQuizConfig && selectedCourse) {
    return (
      <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
        <div className="bg-white rounded-3xl shadow-2xl p-8 max-w-md w-full">
          <div className="flex justify-between items-center mb-6">
            <h2 className="text-2xl font-bold">Set Quiz Duration</h2>
            <button onClick={resetAll} className="text-gray-600 hover:text-gray-800">
              <X className="w-6 h-6" />
            </button>
          </div>
          <p className="text-gray-600 mb-6 text-center">How many minutes?</p>
          <input
            type="number"
            min="1"
            max="120"
            value={quizDuration}
            onChange={(e) => setQuizDuration(Math.max(1, parseInt(e.target.value) || 15))}
            className="w-full p-4 text-center text-4xl font-bold border-2 border-gray-300 rounded-2xl focus:border-indigo-500 outline-none"
            autoFocus
          />
          <div className="mt-8 flex gap-4">
            <button onClick={startQuizMode} className="flex-1 bg-gradient-to-r from-indigo-600 to-purple-600 text-white py-4 rounded-2xl font-semibold flex items-center justify-center gap-3">
              <Play className="w-6 h-6" /> Start Quiz
            </button>
            <button onClick={resetAll} className="flex-1 bg-gray-200 text-gray-700 py-4 rounded-2xl font-semibold">
              Cancel
            </button>
          </div>
        </div>
      </div>
    );
  }

  // === REVIEW MODE ===
  if (isReviewing && questions.length > 0 && selectedCourse) {
    const userAnswer = userAnswers[currentQuestion.id];
    const isCorrect = userAnswer === currentQuestion.answer;
    const wasAnswered = userAnswer !== undefined;

    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-white to-purple-100 py-12 px-4">
        <div className="max-w-4xl mx-auto">
          <div className="flex items-center justify-between mb-8">
            <button onClick={resetAll} className="text-gray-600 hover:text-gray-800">
              <ArrowLeft className="w-8 h-8" />
            </button>
            <div>
              <h1 className="text-3xl font-bold">{selectedCourse.name}</h1>
              <p className="text-gray-600 flex items-center gap-2">
                <Award className="w-5 h-5" /> Review Answers
              </p>
            </div>
            <div />
          </div>

          <div className="mb-6">
            <div className="flex justify-between text-sm text-gray-600 mb-2">
              <span>Question {currentQuestionIndex + 1} of {questions.length}</span>
              <span>
                {wasAnswered 
                  ? (isCorrect ? <span className="text-emerald-600 font-bold">Correct</span> 
                     : <span className="text-red-600 font-bold">Incorrect</span>)
                  : <span className="text-orange-600 font-bold">Unanswered</span>
                }
              </span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-3">
              <div className="bg-gradient-to-r from-indigo-600 to-purple-600 h-3 rounded-full transition-all"
                style={{ width: `${((currentQuestionIndex + 1) / questions.length) * 100}%` }} />
            </div>
          </div>

          <div className="bg-white rounded-3xl shadow-xl p-8 mb-8">
            <h2 className="text-2xl font-semibold mb-8">{currentQuestion.question}</h2>

            <div className="space-y-4">
              {Array.isArray(currentQuestion.options) ? currentQuestion.options.map((option, idx) => {
                const isSelected = userAnswer === option;
                const isCorrectAnswer = option === currentQuestion.answer;

                let btnClass = "w-full p-5 text-left rounded-2xl border-2 cursor-default ";
                if (isCorrectAnswer) {
                  btnClass += "bg-emerald-50 border-emerald-500 text-emerald-800";
                } else if (isSelected && !isCorrectAnswer) {
                  btnClass += "bg-red-50 border-red-500 text-red-800";
                } else if (!wasAnswered) {
                  btnClass += "bg-gray-50 border-gray-300 text-gray-600";
                } else {
                  btnClass += "bg-white border-gray-300 text-gray-700";
                }

                return (
                  <div key={idx} className={btnClass}>
                    <div className="flex items-center gap-4">
                      <div className={`w-8 h-8 rounded-full border-2 flex items-center justify-center font-bold
                        ${isCorrectAnswer ? 'bg-emerald-500 border-emerald-500 text-white' :
                          isSelected && !isCorrectAnswer ? 'bg-red-500 border-red-500 text-white' :
                          'border-gray-400 text-gray-600'}`}>
                        {String.fromCharCode(65 + idx)}
                      </div>
                      <span>{option}</span>
                      {isCorrectAnswer && <CheckCircle className="w-6 h-6 text-emerald-600 ml-auto" />}
                      {isSelected && !isCorrectAnswer && <XCircle className="w-6 h-6 text-red-600 ml-auto" />}
                    </div>
                  </div>
                );
              }) : <p>No options available.</p>}
            </div>

            {currentQuestion.explanation && (
              <div className="mt-8 bg-blue-50 border-l-4 border-blue-500 p-6 rounded-r-2xl">
                <div className="flex gap-3">
                  <AlertCircle className="w-6 h-6 text-blue-600 flex-shrink-0" />
                  <div>
                    <h3 className="font-bold text-blue-800 mb-1">Explanation</h3>
                    <p className="text-blue-700">{currentQuestion.explanation}</p>
                  </div>
                </div>
              </div>
            )}

            {(!wasAnswered || !isCorrect) && (
              <div className="mt-6 p-4 bg-amber-50 border border-amber-300 rounded-2xl">
                <p className="text-amber-800 font-semibold">
                  Correct Answer: <span className="underline">{currentQuestion.answer}</span>
                </p>
              </div>
            )}
          </div>

          <div className="flex justify-between items-center">
            <button
              onClick={previousQuestion}
              disabled={currentQuestionIndex === 0}
              className="px-6 py-3 rounded-2xl bg-gray-200 disabled:opacity-50 flex items-center gap-2"
            >
              <ArrowLeft className="w-5 h-5" /> Previous
            </button>

            {currentQuestionIndex < questions.length - 1 ? (
              <button
                onClick={nextQuestion}
                className="px-8 py-3 rounded-2xl bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-semibold flex items-center gap-2"
              >
                Next <ArrowRight className="w-5 h-5" />
              </button>
            ) : (
              <button
                onClick={resetAll}
                className="px-8 py-3 rounded-2xl bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-semibold"
              >
                Finish Review & Return
              </button>
            )}
          </div>
        </div>
      </div>
    );
  }

  // === RESULTS SCREEN ===
  if (showResults && calculateResults) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-white to-purple-100 py-12 px-4">
        <div className="max-w-4xl mx-auto">
          <div className="bg-white/90 backdrop-blur-md rounded-3xl shadow-2xl p-8 text-center">
            <Award className="w-20 h-20 text-amber-500 mx-auto mb-4" />
            <h1 className="text-4xl font-bold mb-2">Quiz Complete!</h1>
            <p className="text-xl text-gray-600 mb-8">{selectedCourse.name}</p>

            <div className="grid grid-cols-2 md:grid-cols-4 gap-6 mb-10">
              <div className="bg-emerald-50 p-6 rounded-2xl">
                <CheckCircle className="w-8 h-8 text-emerald-600 mx-auto mb-2" />
                <div className="text-3xl font-bold text-emerald-600">{calculateResults.correct}</div>
                <div className="text-sm text-emerald-700">Correct</div>
              </div>
              <div className="bg-red-50 p-6 rounded-2xl">
                <XCircle className="w-8 h-8 text-red-600 mx-auto mb-2" />
                <div className="text-3xl font-bold text-red-600">{calculateResults.wrong}</div>
                <div className="text-sm text-red-700">Wrong</div>
              </div>
              <div className="bg-gray-50 p-6 rounded-2xl">
                <Clock className="w-8 h-8 text-gray-600 mx-auto mb-2" />
                <div className="text-3xl font-bold text-gray-600">{calculateResults.unanswered}</div>
                <div className="text-sm text-gray-700">Unanswered</div>
              </div>
              <div className="bg-indigo-50 p-6 rounded-2xl">
                <Award className="w-8 h-8 text-indigo-600 mx-auto mb-2" />
                <div className="text-3xl font-bold text-indigo-600">{calculateResults.percentage}%</div>
                <div className="text-sm text-indigo-700">Score</div>
              </div>
            </div>

            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={() => {
                  setShowResults(false);
                  setIsReviewing(true);
                  setCurrentQuestionIndex(0);
                }}
                className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white py-3 px-8 rounded-2xl font-semibold hover:from-indigo-700 hover:to-purple-700 flex items-center justify-center gap-2"
              >
                <RotateCcw className="w-5 h-5" />
                Review Answers
              </button>
              <button onClick={resetAll} className="bg-gray-200 text-gray-700 py-3 px-8 rounded-2xl font-semibold hover:bg-gray-300">
                Back to Quizzes
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // === QUESTION SCREEN ===
  if (currentQuestion && currentMode && selectedCourse) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-white to-purple-100 py-12 px-4">
        <div className="max-w-4xl mx-auto">
          <div className="flex items-center justify-between mb-8">
            <button onClick={resetAll} className="text-gray-600 hover:text-gray-800">
              <ArrowLeft className="w-8 h-8" />
            </button>
            <div>
              <h1 className="text-3xl font-bold">{selectedCourse.name}</h1>
              <p className="text-gray-600 flex items-center gap-2 capitalize">
                {currentMode === 'learning' ? <BookOpen className="w-5 h-5" /> : <Award className="w-5 h-5" />}
                {currentMode} Mode
              </p>
            </div>
            {currentMode === 'quiz' && (
              <div className="flex items-center gap-4">
                <button onClick={() => setIsQuizActive(!isQuizActive)} className="bg-white/80 p-2 rounded-full shadow">
                  {isQuizActive ? <Pause className="w-6 h-6" /> : <Play className="w-6 h-6" />}
                </button>
                <div className={`px-4 py-2 rounded-2xl font-mono text-xl font-bold ${timeLeft <= 60 ? 'text-red-600 animate-pulse' : 'text-gray-800'}`}>
                  <Clock className="w-5 h-5 inline mr-2" />
                  {formatTime(timeLeft)}
                </div>
              </div>
            )}
          </div>

          <div className="mb-6">
            <div className="flex justify-between text-sm text-gray-600 mb-2">
              <span>Question {currentQuestionIndex + 1} of {questions.length}</span>
              <span>{Math.round(((currentQuestionIndex + 1) / questions.length) * 100)}% Complete</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-3">
              <div className="bg-gradient-to-r from-indigo-600 to-purple-600 h-3 rounded-full transition-all"
                style={{ width: `${((currentQuestionIndex + 1) / questions.length) * 100}%` }} />
            </div>
          </div>

          <div className="bg-white rounded-3xl shadow-xl p-8 mb-8">
            <h2 className="text-2xl font-semibold mb-8">{currentQuestion.question}</h2>

            <div className="space-y-4">
              {Array.isArray(currentQuestion.options) ? currentQuestion.options.map((option, idx) => {
                const isSelected = userAnswers[currentQuestion.id] === option;
                const isCorrect = option === currentQuestion.answer;
                const showFeedback = showExplanation && currentMode === 'learning';

                let btnClass = "w-full p-5 text-left rounded-2xl border-2 transition-all text-lg ";
                if (showFeedback) {
                  if (isCorrect) btnClass += "bg-emerald-50 border-emerald-500 text-emerald-800";
                  else if (isSelected && !isCorrect) btnClass += "bg-red-50 border-red-500 text-red-800";
                  else btnClass += "bg-gray-50 border-gray-300 text-gray-600";
                } else if (isSelected) {
                  btnClass += "bg-indigo-50 border-indigo-500 scale-105";
                } else {
                  btnClass += "bg-white border-gray-300 hover:bg-gray-50 hover:border-gray-400";
                }

                return (
                  <button
                    key={idx}
                    onClick={() => handleAnswerSelect(currentQuestion.id, option)}
                    disabled={showFeedback}
                    className={btnClass}
                  >
                    <div className="flex items-center gap-4">
                      <div className={`w-8 h-8 rounded-full border-2 flex items-center justify-center font-bold
                        ${showFeedback && isCorrect ? 'bg-emerald-500 border-emerald-500 text-white' :
                          showFeedback && isSelected && !isCorrect ? 'bg-red-500 border-red-500 text-white' :
                          isSelected ? 'bg-indigo-500 border-indigo-500 text-white' : 'border-gray-400'}`}>
                        {String.fromCharCode(65 + idx)}
                      </div>
                      <span>{option}</span>
                      {showFeedback && isCorrect && <CheckCircle className="w-6 h-6 text-emerald-600 ml-auto" />}
                      {showFeedback && isSelected && !isCorrect && <XCircle className="w-6 h-6 text-red-600 ml-auto" />}
                    </div>
                  </button>
                );
              }) : <p className="text-red-600">No options available.</p>}
            </div>

            {showExplanation && currentMode === 'learning' && currentQuestion.explanation && (
              <div className="mt-8 bg-blue-50 border-l-4 border-blue-500 p-6 rounded-r-2xl">
                <div className="flex gap-3">
                  <AlertCircle className="w-6 h-6 text-blue-600 flex-shrink-0" />
                  <div>
                    <h3 className="font-bold text-blue-800 mb-1">Explanation</h3>
                    <p className="text-blue-700">{currentQuestion.explanation}</p>
                  </div>
                </div>
              </div>
            )}
          </div>

          <div className="flex justify-between">
            <button
              onClick={previousQuestion}
              disabled={currentQuestionIndex === 0}
              className="px-6 py-3 rounded-2xl bg-gray-200 disabled:opacity-50 flex items-center gap-2"
            >
              <ArrowLeft className="w-5 h-5" /> Previous
            </button>

            <button
              onClick={nextQuestion}
              disabled={!userAnswers[currentQuestion.id] && currentMode === 'learning'}
              className="px-8 py-3 rounded-2xl bg-gradient-to-r from-indigo-600 to-purple-600 text-white font-semibold disabled:opacity-50 flex items-center gap-2"
            >
              {currentQuestionIndex === questions.length - 1
                ? (currentMode === 'quiz' ? 'Finish Quiz' : 'Complete')
                : 'Next'}
              <ArrowRight className="w-5 h-5" />
            </button>
          </div>
        </div>
      </div>
    );
  }

  // === MAIN LIST VIEW ===
  return (
    <RequirePremium>
    <div className="min-h-screen bg-gradient-to-br  py-12 px-0">
     <div className="max-w-7xl mx-auto">
  <div className="text-center mb-12">
    <h1 className="text-5xl font-bold text-gray-900 mb-4">
      Public Quizzes
    </h1>
    <p className="text-xl text-gray-600 max-w-3xl mx-auto">
      Explore community quizzes with Learning or Timed Quiz modes.
    </p>
  </div>

  <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-8 mb-10">
    <div className="flex flex-wrap gap-6 items-end">
      <div className="flex-1 min-w-[250px]">
        <label className="block text-sm font-semibold text-gray-900 mb-2">Search</label>
        <div className="relative">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
          <input
            type="text"
            onChange={(e) => debouncedSetSearchQuery(e.target.value)}
            placeholder="Search by name..."
            className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none"
          />
        </div>
      </div>

      <div>
        <label className="block text-sm font-semibold text-gray-900 mb-2">Subject</label>
        <select
          value={selectedSubject}
          onChange={(e) => setSelectedSubject(e.target.value)}
          disabled={loading.subjects}
          className="px-4 py-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none disabled:opacity-50 bg-white"
        >
          <option value="all">All Subjects</option>
          {subjects.map((subj) => (
            <option key={subj.id} value={subj.id}>{subj.name}</option>
          ))}
        </select>
      </div>

      <div>
        <label className="block text-sm font-semibold text-gray-900 mb-2">Subtopic</label>
        <select
          value={selectedSubtopic}
          onChange={(e) => setSelectedSubtopic(e.target.value)}
          disabled={selectedSubject === "all" || loading.subtopics}
          className="px-4 py-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none disabled:opacity-50 bg-white"
        >
          <option value="all">All Subtopics</option>
          {subtopics.map((sub) => (
            <option key={sub.id} value={sub.id}>{sub.name}</option>
          ))}
        </select>
      </div>
    </div>
  </div>

  {/* Loading / Error / Empty */}
  {loading.courses ? (
    <div className="flex justify-center items-center py-20">
      <Loader className="w-10 h-10 text-indigo-600 animate-spin" />
      <span className="ml-4 text-lg text-gray-600">Loading quizzes...</span>
    </div>
  ) : error ? (
    <div className="text-center py-20">
      <AlertCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
      <p className="text-xl text-red-600">{error}</p>
    </div>
  ) : courses.length === 0 ? (
    <div className="text-center py-20">
      <BookOpen className="w-20 h-20 text-gray-400 mx-auto mb-6" />
      <h3 className="text-2xl font-semibold text-gray-700 mb-2">No quizzes found</h3>
      <p className="text-gray-500">Try adjusting your filters.</p>
    </div>
  ) : (
    <>
      {/* Course Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8 mb-16">
  {courses.map((course) => (
    <article
      key={course.id}
      onClick={() => handleCourseClick(course)}
      className="group cursor-pointer bg-white rounded-2xl shadow-md hover:shadow-xl transition-all duration-400 border border-gray-200 hover:border-gray-300 overflow-hidden"
    >
      {/* Subtle Top Accent Line */}
      {/* <div className="h-1 bg-gradient-to-r from-indigo-600 to-indigo-700"></div> */}

      {/* Main Content */}
      <div className="p-7">
        {/* Quiz Title */}
        <h3 className="text-xl font-semibold text-gray-900 mb-4 line-clamp-2 leading-tight group-hover:text-indigo-700 transition-colors">
          {course.name}
        </h3>

        {/* Subject & Subtopic - Clean Tag Style */}
        <div className="flex flex-wrap gap-3 mb-6">
          <span className="inline-flex items-center gap-2 px-4 py-1.5 bg-indigo-50 text-indigo-800 rounded-full text-sm font-medium">
            <BookOpen className="w-4 h-4" />
            {course.subject || "General"}
          </span>
          {course.subtopic && (
            <span className="inline-flex items-center px-4 py-1.5 bg-gray-100 text-gray-700 rounded-full text-sm font-medium">
              {course.subtopic}
            </span>
          )}
        </div>

        {/* Metadata Row */}
        <div className="flex items-center justify-between text-gray-600 mb-8">
          <div className="flex items-center gap-2.5">
            <Users className="w-5 h-5 text-indigo-600" />
            <span className="font-medium text-base">
              {course.personal_questions_count || 0} Questions
            </span>
          </div>
          <div className="text-sm flex items-center gap-2">
            <Clock className="w-4 h-4" />
            <time dateTime={course.created_at}>
              {new Date(course.created_at || Date.now()).toLocaleDateString('en-US', {
                month: 'short',
                day: 'numeric',
                year: 'numeric',
              })}
            </time>
          </div>
        </div>

        {/* Call to Action - Professional Button */}
        <button className="w-full bg-gradient-to-r from-indigo-600 to-indigo-700 hover:from-indigo-700 hover:to-indigo-800 text-white font-semibold py-3.5 px-6 rounded-xl flex items-center justify-center gap-3 transition-all duration-300 shadow-sm group-hover:shadow-md">
          <span>Start Quiz</span>
          <ChevronRight className="w-5 h-5 transition-transform group-hover:translate-x-1" />
        </button>
      </div>
    </article>
  ))}
</div>

      {/* Pagination Controls */}
      {totalPages > 1 && (
        <div className="flex flex-col sm:flex-row items-center justify-between gap-6 bg-white rounded-2xl shadow-sm border border-gray-200 p-6">
          <div className="text-gray-600 text-sm font-medium">
            Showing {(currentPage - 1) * perPage + 1} to{' '}
            {Math.min(currentPage * perPage, totalCourses)} of {totalCourses} quizzes
          </div>

          <div className="flex items-center gap-3">
            <button
              onClick={() => goToPage(currentPage - 1)}
              disabled={currentPage === 1}
              className="p-3 rounded-lg border border-gray-300 bg-white disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50 transition"
            >
              <ChevronLeft className="w-5 h-5" />
            </button>

            <div className="flex gap-2">
              {renderPageNumbers()}
            </div>

            <button
              onClick={() => goToPage(currentPage + 1)}
              disabled={currentPage === totalPages}
              className="p-3 rounded-lg border border-gray-300 bg-white disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-50 transition"
            >
              <ChevronRightIcon className="w-5 h-5" />
            </button>
          </div>
        </div>
      )}
    </>
  )}

  {/* Mode Selection Modal */}
  {showModeModal && selectedCourse && (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-2xl shadow-xl border border-gray-200 p-8 max-w-md w-full">
        {/* Header */}
        <div className="flex justify-between items-center mb-6">
          <h2 className="text-2xl font-bold text-gray-900">
            {selectedCourse.name}
          </h2>
          <button
            onClick={resetAll}
            className="text-gray-500 hover:text-gray-700 transition-colors"
            aria-label="Close modal"
          >
            <X className="w-6 h-6" />
          </button>
        </div>
    
        {/* Description */}
        <p className="text-gray-600 mb-8 text-lg">Choose your mode:</p>
    
        {/* Mode Buttons */}
        <div className="space-y-4">
          {/* Primary: Learning Mode */}
          <button
            onClick={() => handleModeSelect('learning')}
            className="w-full bg-indigo-600 hover:bg-indigo-700 text-white py-4 rounded-lg font-semibold flex items-center justify-center gap-3 transition-colors duration-200 shadow-sm"
          >
            <BookOpen className="w-6 h-6" />
            Learning Mode
          </button>
    
          {/* Secondary: Timed Quiz Mode */}
          <button
            onClick={() => handleModeSelect('quiz')}
            className="w-full border-2 border-gray-300 hover:border-gray-400 text-gray-800 py-4 rounded-lg font-semibold flex items-center justify-center gap-3 transition-all duration-200 hover:bg-gray-50"
          >
            <Award className="w-6 h-6" />
            Timed Quiz Mode
          </button>
        </div>
      </div>
    </div>
  )}
</div>
    </div>
    </RequirePremium>
  );
};

export default PublicQuiz;