import React, { useState, useEffect, createContext, useContext } from 'react';
import { useNavigate, useLocation, Outlet } from 'react-router-dom';
import { User, Stethoscope, BookOpen, Upload, Eye, Brain, Menu, X, Bell, Settings, LogOut, ChevronDown, Home, FileText, Zap, GraduationCap, Globe, Crown, FolderOpen, Shield, Heart, Mail, Copyright } from 'lucide-react';
import { usePremium } from '../../context/premiumContext';
import TrialMarquee from './TrialMarquee';

// Create a context for sharing stats and profile data
const StatsContext = createContext();
export const useStats = () => useContext(StatsContext);

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;

const Loader = () => (
  <div className="fixed inset-0 z-50 flex items-center justify-center bg-white">
    <div className="flex flex-col items-center space-y-4">
      <div className="relative">
        <div className="w-16 h-16 border-4 border-t-indigo-600 border-gray-200 rounded-full animate-spin"></div>
      </div>
      <p className="text-lg font-semibold text-gray-700">Loading MedArchieve...</p>
    </div>
  </div>
);

const MedicalStudentDashboard = () => {
  const navigate = useNavigate();
  const location = useLocation();
  const { refreshPremiumStatus } = usePremium();
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [profileDropdownOpen, setProfileDropdownOpen] = useState(false);
  const [loading, setLoading] = useState({ stats: false, profile: false });
  const [isLoggingOut, setIsLoggingOut] = useState(false);
  const [pageLoading, setPageLoading] = useState(true);
  const [stats, setStats] = useState({
    questionsUploaded: 0,
    questionsAnswered: 0,
    studyStreak: 0,
    totalPoints: 0,
  });
  const [recentActivity, setRecentActivity] = useState([]);
  const [achievements, setAchievements] = useState([]);

  const [userProfile, setUserProfile] = useState({
    name: '',
    email: '',
    year: '',
    university: '',
    bio: 'Medical student passionate about learning and sharing knowledge...',
    location: '',
    role: 'user',
    institution: null,
  });

  // Get active tab from current route
  const getActiveTab = () => {
    const path = location.pathname.split('/').pop();
    return path === 'dashboard' ? 'dashboard' : path;
  };

  const navigationItems = [
    { icon: <Home size={20} />, label: 'Dashboard', path: '/dashboard' },
    { icon: <User size={20} />, label: 'Profile', path: '/dashboard/profile' },
    { icon: <Brain size={20} />, label: 'Quiz', path: '/dashboard/quiz' },
    { icon: <Zap size={20} />, label: 'Quiz Generator', path: '/dashboard/generator' },
    { icon: <GraduationCap size={20} />, label: 'Courses', path: '/dashboard/courses' },
    { icon: <Globe size={20} />, label: 'Public Quiz', path: '/dashboard/public-quiz' },
    { icon: <Globe size={20} />, label: 'FlashCard Hub', path: '/dashboard/flashcards' },
    { icon: <Crown size={20} />, label: 'Upgrade to Premium', path: '/dashboard/premium' },
    ...(userProfile.role === 'admin' || userProfile.role === 'superadmin'
      ? [
          { icon: <Shield size={20} />, label: 'Admin Dashboard', path: '/dashboard/admin' },
          { icon: <FolderOpen size={20} />, label: 'Manage Subjects', path: '/dashboard/manage-subjects' }
        ]
      : []),
  ];

  // Premium status real-time refresh
  useEffect(() => {
    const handleFocus = () => {
      refreshPremiumStatus();
    };

    window.addEventListener('focus', handleFocus);
    refreshPremiumStatus(); // Initial call

    return () => {
      window.removeEventListener('focus', handleFocus);
    };
  }, [refreshPremiumStatus]);

  useEffect(() => {
    const token = localStorage.getItem('auth_token');
    if (!token) {
      navigate('/register');
      return;
    }

    // Load profile from localStorage for sidebar display
    const loadProfileFromStorage = () => {
      const storedName = localStorage.getItem('user_name') || '';
      const storedEmail = localStorage.getItem('user_email') || '';
      const storedYear = localStorage.getItem('user_year') || '';
      const storedUniversity = localStorage.getItem('user_university') || '';
      const storedRole = localStorage.getItem('user_role') || 'user';
      
      setUserProfile(prev => ({
        ...prev,
        name: storedName,
        email: storedEmail,
        year: storedYear,
        university: storedUniversity,
        role: storedRole,
      }));
    };

    // Load profile data from localStorage immediately
    loadProfileFromStorage();

    let timer;
    const fetchProfile = async () => {
      setLoading((prev) => ({ ...prev, profile: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/profile`, {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
            Accept: 'application/json',
          },
        });

        if (response.status === 401) {
          localStorage.removeItem('auth_token');
          navigate('/register');
          return;
        }

        if (!response.ok) {
          throw new Error(`HTTP error! Status: ${response.status}`);
        }

        const data = await response.json();
        const profileData = data.data || data || {};
        const profile = {
          name: profileData.name || '',
          email: profileData.email || '',
          year: profileData.year || '',
          university: profileData.university || '',
          bio: profileData.bio || 'Medical student passionate about learning and sharing knowledge...',
          location: profileData.location || '',
          role: profileData.role || 'user',
          institution: profileData.institution || null,
        };
        
        // Update localStorage with fresh data
        localStorage.setItem('user_name', profile.name);
        localStorage.setItem('user_email', profile.email);
        localStorage.setItem('user_year', profile.year);
        localStorage.setItem('user_university', profile.university);
        localStorage.setItem('user_role', profile.role);
        
        setUserProfile(profile);
      } catch (err) {
        console.error('Error fetching profile:', err);
        if (err.message.includes('401')) {
          localStorage.removeItem('auth_token');
          navigate('/register');
        }
      } finally {
        setLoading((prev) => ({ ...prev, profile: false }));
      }
    };

    const fetchStatsAndActivity = async () => {
      setLoading((prev) => ({ ...prev, stats: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/user-stats`, {
          headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
            Accept: 'application/json',
          },
        });

        if (response.status === 401) {
          localStorage.removeItem('auth_token');
          navigate('/register');
          return;
        }

        if (!response.ok) {
          throw new Error(`HTTP error! Status: ${response.status}`);
        }

        const data = await response.json();
        setStats({
          questionsUploaded: data.stats?.questionsUploaded || 0,
          questionsAnswered: data.stats?.questionsAnswered || 0,
          studyStreak: data.stats?.studyStreak || 0,
          totalPoints: data.stats?.totalPoints || 0,
        });
        setRecentActivity(data.activity || []);
        setAchievements(data.achievements || []);
      } catch (err) {
        console.error('Error fetching stats:', err);
        if (err.message.includes('401')) {
          localStorage.removeItem('auth_token');
          navigate('/register');
        }
      } finally {
        setLoading((prev) => ({ ...prev, stats: false }));
      }
    };

    const startLoading = async () => {
      const startTime = Date.now();
      await Promise.all([fetchProfile(), fetchStatsAndActivity()]);
      const elapsedTime = Date.now() - startTime;
      const remainingTime = 2000 - elapsedTime;
      if (remainingTime > 0) {
        timer = setTimeout(() => setPageLoading(false), remainingTime);
      } else {
        setPageLoading(false);
      }
    };

    startLoading();

    return () => clearTimeout(timer);
  }, [navigate]);

  const handleLogout = async () => {
    const token = localStorage.getItem('auth_token');
    if (!token) {
      navigate('/register');
      return;
    }

    setIsLoggingOut(true);

    try {
      const response = await fetch(`${API_BASE_URL}/logout`, {
        method: 'POST',
        headers: {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
          Accept: 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! Status: ${response.status}`);
      }

      localStorage.removeItem('auth_token');
      localStorage.removeItem('user_name');
      localStorage.removeItem('user_email');
      localStorage.removeItem('user_year');
      localStorage.removeItem('user_university');
      localStorage.removeItem('user_role');
      setProfileDropdownOpen(false);
      navigate('/register');
    } catch (err) {
      console.error('Error during logout:', err);
      localStorage.removeItem('auth_token');
      localStorage.removeItem('user_name');
      localStorage.removeItem('user_email');
      localStorage.removeItem('user_year');
      localStorage.removeItem('user_university');
      localStorage.removeItem('user_role');
      navigate('/register');
    } finally {
      setIsLoggingOut(false);
    }
  };

  const Sidebar = () => (
    <>
      {sidebarOpen && (
        <div
          className="fixed inset-0 z-40 bg-black bg-opacity-50 lg:hidden"
          onClick={() => setSidebarOpen(false)}
        />
      )}
      
      <div
        className={`fixed inset-y-0 left-0 z-50 w-64 bg-white border-r border-gray-200 transform ${
          sidebarOpen ? 'translate-x-0' : '-translate-x-full'
        } transition-transform duration-300 ease-in-out lg:translate-x-0 lg:static lg:inset-0 flex flex-col`}
      >
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200 flex-shrink-0">
          <div className="flex items-center space-x-3">
            <div className="w-8 h-8 bg-[#38bdf8] rounded-lg flex items-center justify-center">
              <Stethoscope size={20} className="text-white" />
            </div>
            <h2 className="text-xl font-bold text-[#1e3a8a]">
              MedArchieve
            </h2>
          </div>
          <button
            onClick={() => setSidebarOpen(false)}
            className="lg:hidden p-2 rounded-lg hover:bg-gray-100 transition-colors"
          >
            <X size={20} className="text-gray-600" />
          </button>
        </div>
  
        {/* User Profile */}
        <div className="p-6 border-b border-gray-200 flex-shrink-0">
          <div className="flex items-center space-x-3">
            <div className="w-10 h-10 bg-indigo-700 rounded-full flex items-center justify-center">
              <User size={20} className="text-white" />
            </div>
            <div className="flex-1 min-w-0">
              <p className="text-sm font-semibold text-gray-900 truncate">{userProfile.name}</p>
              <p className="text-xs text-gray-600 truncate">{userProfile.year} Level</p>
            </div>
          </div>
          <div className="mt-3 text-xs text-gray-600">
            <p className="truncate">{userProfile.university}</p>
          </div>
        </div>
  
        {/* Navigation */}
        <nav className="flex-1 overflow-y-auto mt-2 px-3 pb-6">
          <div className="space-y-1">
            {navigationItems.map((item) => (
              <NavItem
                key={item.path}
                icon={item.icon}
                label={item.label}
                path={item.path}
                activeTab={getActiveTab()}
                setSidebarOpen={setSidebarOpen}
              />
            ))}
          </div>
        </nav>
      </div>
    </>
  );

  const NavItem = ({ icon, label, path, activeTab, setSidebarOpen }) => {
    const navigate = useNavigate();
    const pathKey = path.split('/').pop();
    const isActive = activeTab === pathKey || (path === '/dashboard' && activeTab === 'dashboard');
    
    return (
      <button
        onClick={() => {
          navigate(path);
          setSidebarOpen(false);
        }}
        className={`w-full flex items-center space-x-3 px-3 py-2.5 text-left rounded-lg transition-all duration-200 ${
          isActive
            ? 'bg-indigo-50 text-indigo-700 font-medium'
            : 'text-gray-700 hover:bg-gray-50 hover:text-gray-900'
        }`}
      >
        <div className={`${isActive ? 'text-indigo-700' : 'text-gray-500'}`}>
          {icon}
        </div>
        <span className="text-sm">{label}</span>
        {isActive && (
          <div className="ml-auto w-1.5 h-1.5 bg-indigo-600 rounded-full"></div>
        )}
      </button>
    );
  };

  const Header = () => (
    <header className="bg-white border-b border-gray-200 relative z-30">
      {/* Trial Marquee - Full width, above header content */}
      <TrialMarquee />
  
      {/* Main Header Content */}
      <div className="px-4 py-4 lg:px-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <button
              onClick={() => setSidebarOpen(true)}
              className="lg:hidden p-2 rounded-lg hover:bg-gray-100 transition-colors"
            >
              <Menu size={24} className="text-gray-600" />
            </button>
            <div>
              <h1 className="text-xl font-semibold text-gray-900 capitalize">
                {getActiveTab() === 'dashboard' ? 'Dashboard' : getActiveTab().replace(/([A-Z])/g, ' $1').trim()}
              </h1>
              <p className="text-sm text-gray-500 mt-0.5 hidden sm:block">
                {new Date().toLocaleDateString('en-US', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}
              </p>
            </div>
          </div>
  
          <div className="flex items-center space-x-3">
            <button className="p-2 rounded-lg hover:bg-gray-100 transition-colors relative">
              <Bell size={20} className="text-gray-600" />
              <div className="absolute top-1.5 right-1.5 w-2 h-2 bg-red-500 rounded-full border-2 border-white"></div>
            </button>
  
            <div className="relative">
              <button
                onClick={() => setProfileDropdownOpen(!profileDropdownOpen)}
                className="flex items-center space-x-2 p-2 rounded-lg hover:bg-gray-100 transition-colors"
              >
                <div className="w-8 h-8 bg-indigo-600 rounded-full flex items-center justify-center">
                  <User className="text-white" size={16} />
                </div>
                <ChevronDown
                  size={16}
                  className={`text-gray-600 transition-transform hidden sm:block ${
                    profileDropdownOpen ? 'rotate-180' : ''
                  }`}
                />
              </button>
  
              {profileDropdownOpen && (
                <div className="absolute right-0 mt-2 w-56 bg-white rounded-lg shadow-lg border border-gray-200 py-1 z-50">
                  <div className="px-4 py-3 border-b border-gray-100">
                    <p className="text-sm font-semibold text-gray-900 truncate">{userProfile.name}</p>
                    <p className="text-xs text-gray-600 truncate mt-0.5">{userProfile.email}</p>
                  </div>
                  <button
                    onClick={() => {
                      navigate('/dashboard/profile');
                      setProfileDropdownOpen(false);
                    }}
                    className="w-full flex items-center space-x-3 px-4 py-2.5 text-left text-sm text-gray-700 hover:bg-gray-50 transition-colors"
                  >
                    <User size={16} className="text-gray-500" />
                    <span>View Profile</span>
                  </button>
                  <hr className="my-1 border-gray-100" />
                  <button
                    onClick={handleLogout}
                    disabled={isLoggingOut}
                    className={`w-full flex items-center space-x-3 px-4 py-2.5 text-left text-sm transition-colors ${
                      isLoggingOut 
                        ? 'text-red-400 bg-red-50 cursor-not-allowed opacity-60' 
                        : 'text-red-600 hover:bg-red-50'
                    }`}
                  >
                    <LogOut size={16} className={isLoggingOut ? 'animate-spin' : ''} />
                    <span>{isLoggingOut ? 'Signing out...' : 'Sign out'}</span>
                  </button>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </header>
  );

  const Footer = () => (
    <footer className="bg-white border-t border-gray-200 px-4 py-6 lg:px-6 mt-auto">
      <div className="max-w-7xl mx-auto">
        <div className="flex flex-col md:flex-row items-center justify-between gap-4">
          <div className="flex items-center space-x-2 text-gray-600">
            <Copyright size={16} />
            <span className="text-sm">2025 MedStudent. All rights reserved.</span>
          </div>
          
          <div className="text-sm text-gray-500">
            Made with <span className="text-red-500">❤️</span> for medical students
          </div>
        </div>
      </div>
    </footer>
  );

  return (
    <StatsContext.Provider value={{ stats, recentActivity, achievements, loading, userProfile }}>
      {pageLoading ? (
        <Loader />
      ) : (
        <div className="flex h-screen bg-gray-50 flex-col">
          <div className="flex flex-1 overflow-hidden">
            <Sidebar />
            <div className="flex-1 flex flex-col overflow-hidden">
              <Header />
              <main className="flex-1 overflow-y-auto p-4 lg:p-6">
                <Outlet />
              </main>
              <Footer />
            </div>
          </div>
        </div>
      )}
    </StatsContext.Provider>
  );
};

export default MedicalStudentDashboard;