import React, { useState, useEffect, useRef } from 'react';
import { Star, Quote, ChevronLeft, ChevronRight, User } from 'lucide-react';

import userImage from '../assets/user.jpg';

const testimonials = [
  {
    id: 1,
    name: 'Gbenga Olajide Olonisakin',
    quote: 'MedArchieve transformed my study routine! The practice tests were spot-on with real exam questions, and the AI-powered recommendations helped me focus on my weak areas. I passed my USMLE Step 1 with a score I never thought possible.',
    role: 'Medical Student',
    university: 'Ibrahim Badamasi Babangida University',
    avatar: '/api/placeholder/60/60',
    rating: 5,
    exam: '300 level',
    score: '245'
  },
  {
    id: 2,
    name: 'James Patel',
    quote: 'The curated resources and progress tracking helped me stay organized and focused throughout my residency preparation. The community features connected me with peers who shared valuable insights and study strategies.',
    role: 'Resident Physician',
    university: 'Johns Hopkins',
    avatar: '/api/placeholder/60/60',
    rating: 5,
    exam: 'Internal Medicine Board',
    score: '98th percentile'
  },
  {
    id: 3,
    name: 'Emily Chen',
    quote: 'Thanks to MedArchieve, I aced my licensing exams. The study tools are intuitive and effective. The mobile app allowed me to study during my commute and clinical rotations. Absolutely game-changing!',
    role: 'Medical Student',
    university: 'Stanford Medicine',
    avatar: '/api/placeholder/60/60',
    rating: 5,
    exam: 'COMLEX Level 1',
    score: '650+'
  },
  {
    id: 4,
    name: 'Dr. Michael Rodriguez',
    quote: 'As someone who struggled with traditional study methods, MedArchieve\'s visual learning approach and spaced repetition system made complex topics finally click. The detailed analytics showed exactly where I needed improvement.',
    role: 'Resident Physician',
    university: 'Mayo Clinic',
    avatar: '/api/placeholder/60/60',
    rating: 5,
    exam: 'USMLE Step 2 CK',
    score: '255'
  },
  {
    id: 5,
    name: 'Dr. Aisha Okonkwo',
    quote: 'The collaborative study features and expert-curated content made my board preparation efficient and stress-free. I particularly loved the real-time feedback and personalized study plans that adapted to my learning pace.',
    role: 'Attending Physician',
    university: 'University of Pennsylvania',
    avatar: '/api/placeholder/60/60',
    rating: 5,
    exam: 'Board Certification',
    score: 'Pass with Distinction'
  }
];

function Testimonials() {
  const [currentSlide, setCurrentSlide] = useState(0);
  const [isVisible, setIsVisible] = useState(false);
  const [isPaused, setIsPaused] = useState(false);
  const [counts, setCounts] = useState([0, 0, 0, 0]); // Initialize counters
  const sectionRef = useRef(null);
  const autoPlayRef = useRef(null);

  // Intersection Observer for animations
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true);
        } else {
          setIsVisible(false); // Reset visibility to allow re-animation
        }
      },
      { threshold: 0.1 }
    );

    if (sectionRef.current) {
      observer.observe(sectionRef.current);
    }

    return () => observer.disconnect();
  }, []);

  // Counter animation logic
  useEffect(() => {
    if (isVisible) {
      const targetCounts = [500, 95, 4.9, 20]; // Target values: Success Stories, Pass Rate, User Rating, Medical Schools
      const duration = 2000; // Animation duration in ms
      const steps = 60; // Number of animation steps
      const increments = targetCounts.map(target => target / steps);

      // Reset counts to 0 when animation starts
      setCounts([0, 0, 0, 0]);

      const counterInterval = setInterval(() => {
        setCounts(prevCounts =>
          prevCounts.map((count, index) => {
            const target = targetCounts[index];
            const increment = increments[index];
            const newCount = Number(count) + increment; // Ensure count is a number
            // Handle User Rating (index 2) with decimal precision
            if (index === 2) {
              return Math.min(Number(newCount.toFixed(1)), target);
            }
            // Handle other counters as integers
            return Math.min(Math.floor(newCount), target);
          })
        );
      }, duration / steps);

      return () => clearInterval(counterInterval);
    }
  }, [isVisible]);

  // Auto-play functionality
  useEffect(() => {
    if (!isPaused) {
      autoPlayRef.current = setInterval(() => {
        setCurrentSlide((prev) => (prev + 1) % testimonials.length);
      }, 5000);
    }

    return () => {
      if (autoPlayRef.current) {
        clearInterval(autoPlayRef.current);
      }
    };
  }, [isPaused]);

  const nextSlide = () => {
    setCurrentSlide((prev) => (prev + 1) % testimonials.length);
  };

  const prevSlide = () => {
    setCurrentSlide((prev) => (prev - 1 + testimonials.length) % testimonials.length);
  };

  const goToSlide = (index) => {
    setCurrentSlide(index);
  };

  return (
    <section 
      ref={sectionRef}
      className="relative bg-white py-20 lg:py-32 overflow-hidden"
      onMouseEnter={() => setIsPaused(true)}
      onMouseLeave={() => setIsPaused(false)}
    >
      {/* Background Decorative Elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute -top-24 -right-24 w-48 h-48 bg-gradient-to-br from-blue-200 to-purple-200 rounded-full opacity-20 animate-pulse" />
        <div className="absolute -bottom-32 -left-32 w-64 h-64 bg-gradient-to-tr from-cyan-200 to-blue-200 rounded-full opacity-20 animate-pulse" style={{ animationDelay: '2s' }} />
        
        {/* Floating Quote Icons */}
        {[...Array(6)].map((_, i) => (
          <div
            key={i}
            className="absolute opacity-5 animate-float"
            style={{
              left: `${15 + (i * 15)}%`,
              top: `${10 + (i * 12)}%`,
              animationDelay: `${i * 0.8}s`,
              animationDuration: `${4 + (i % 2)}s`
            }}
          >
            <Quote className="w-8 h-8 text-blue-400" />
          </div>
        ))}
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
        
        {/* Header Section */}
        <div className={`text-center mb-16 lg:mb-20 transform transition-all duration-1000 ${
          isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
        }`}>
          <div className="inline-flex items-center px-4 py-2 bg-white/80 backdrop-blur-sm rounded-full border border-blue-200 shadow-sm mb-6">
            <div className="flex -space-x-2 mr-3">
              {[...Array(3)].map((_, i) => (
                <div key={i} className="w-6 h-6 bg-gradient-to-br from-blue-400 to-purple-400 rounded-full border-2 border-white" />
              ))}
            </div>
            <span className="text-sm font-medium text-blue-700">{counts[0]}+ Success Stories</span>
          </div>
          
          <h2 className="text-4xl sm:text-5xl lg:text-6xl font-bold text-gray-900 mb-6">
            <span className="block">What Our</span>
            <span className="bg-gradient-to-r from-blue-700 to-cyan-700 bg-clip-text text-transparent">
              Students Say
            </span>
          </h2>
          
          <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Real stories from medical students and professionals who transformed their careers with MedArchieve
          </p>
        </div>

        {/* Testimonials Carousel */}
        <div className="relative max-w-6xl mx-auto">
          
          {/* Navigation Arrows */}
          <button 
            onClick={prevSlide}
            className="absolute left-0 top-1/2 -translate-y-1/2 -translate-x-4 z-10 w-12 h-12 bg-white/90 backdrop-blur-sm rounded-full shadow-lg border border-gray-200 flex items-center justify-center text-gray-600 hover:text-blue-600 hover:bg-white transition-all duration-300 hover:scale-110"
            aria-label="Previous testimonial"
          >
            <ChevronLeft className="w-6 h-6" />
          </button>
          
          <button 
            onClick={nextSlide}
            className="absolute right-0 top-1/2 -translate-y-1/2 translate-x-4 z-10 w-12 h-12 bg-white/90 backdrop-blur-sm rounded-full shadow-lg border border-gray-200 flex items-center justify-center text-gray-600 hover:text-blue-600 hover:bg-white transition-all duration-300 hover:scale-110"
            aria-label="Next testimonial"
          >
            <ChevronRight className="w-6 h-6" />
          </button>

          {/* Testimonials Grid */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 lg:gap-12">
            {[currentSlide, (currentSlide + 1) % testimonials.length].map((index, cardIndex) => {
              const testimonial = testimonials[index];
              return (
                <div
                  key={testimonial.id}
                  className={`transform transition-all duration-700 ${
                    isVisible ? 'translate-y-0 opacity-100' : 'translate-y-20 opacity-0'
                  }`}
                  style={{ transitionDelay: `${cardIndex * 0.2}s` }}
                >
                  <div className="group relative bg-white/80 backdrop-blur-sm rounded-3xl p-8 lg:p-10 border border-white/50 shadow-xl hover:shadow-2xl transition-all duration-500 hover:scale-105 hover:-translate-y-2">
                    
                    {/* Quote Icon */}
                    <div className="absolute -top-4 -left-4 w-12 h-12 bg-gradient-to-br from-blue-500 to-purple-600 rounded-2xl flex items-center justify-center shadow-lg group-hover:scale-110 transition-transform duration-300">
                      <Quote className="w-6 h-6 text-white" />
                    </div>

                    {/* Star Rating */}
                    <div className="flex items-center justify-between mb-6">
                      <div className="flex space-x-1">
                        {[...Array(testimonial.rating)].map((_, i) => (
                          <Star key={i} className="w-5 h-5 text-yellow-400 fill-current" />
                        ))}
                      </div>
                      <div className="text-sm font-semibold text-blue-600 bg-blue-100 px-3 py-1 rounded-full">
                        {testimonial.exam}
                      </div>
                    </div>

                    {/* Quote Text */}
                    <blockquote className="text-gray-700 text-lg leading-relaxed mb-8 italic">
                      "{testimonial.quote}"
                    </blockquote>

                    {/* Author Info */}
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-4">
                        <div className="relative">
                          <div className="flex items-center justify-center">
                            <img
                              src={userImage}
                              alt="Medical student studying with digital tools"
                              className="w-14 h-14 rounded-full shadow-lg"
                            />
                          </div>
                          <div className="absolute -bottom-1 -right-1 w-5 h-5 bg-green-400 rounded-full border-2 border-white" />
                        </div>
                        <div>
                          <h3 className="font-bold text-gray-900 text-lg">{testimonial.name}</h3>
                          <p className="text-gray-600 text-sm">{testimonial.role}</p>
                          <p className="text-blue-600 text-xs font-medium">{testimonial.university}</p>
                        </div>
                      </div>
                    </div>

                    {/* Hover Effect Elements */}
                    <div className="absolute inset-0 bg-gradient-to-r from-blue-500/0 to-purple-500/0 group-hover:from-blue-500/5 group-hover:to-purple-500/5 rounded-3xl transition-all duration-500" />
                  </div>
                </div>
              );
            })}
          </div>

          {/* Pagination Dots */}
          <div className="flex justify-center space-x-2 mt-12">
            {testimonials.map((_, index) => (
              <button
                key={index}
                onClick={() => goToSlide(index)}
                className={`w-3 h-3 rounded-full transition-all duration-300 ${
                  index === currentSlide || index === (currentSlide + 1) % testimonials.length
                    ? 'bg-blue-600 w-8' 
                    : 'bg-gray-300 hover:bg-blue-400'
                }`}
                aria-label={`Go to testimonial ${index + 1}`}
              />
            ))}
          </div>
        </div>

        {/* Stats Section with Counters */}
        <div className={`grid grid-cols-2 lg:grid-cols-3 gap-8 mt-20 transform transition-all duration-1000 ${
          isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
        }`} style={{ transitionDelay: '0.8s' }}>
          {[
            { number: counts[0], label: 'Success Stories', suffix: '+' },
            { number: counts[1], label: 'Pass Rate', suffix: '%' },
            { number: counts[2], label: 'User Rating', suffix: '/5' },
       
          ].map((stat, index) => (
            <div key={index} className="text-center">
              <div className="text-3xl lg:text-4xl font-bold text-gray-900 mb-2">
                {stat.number}{stat.suffix}
              </div>
              <div className="text-gray-600">{stat.label}</div>
            </div>
          ))}
        </div>

        {/* CTA Section */}
        <div className={`text-center mt-16 transform transition-all duration-1000 ${
          isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
        }`} style={{ transitionDelay: '1s' }}>
          <div className="bg-white/80 backdrop-blur-sm rounded-2xl p-8 border border-white/50 shadow-xl max-w-2xl mx-auto">
            <h3 className="text-2xl font-bold text-gray-900 mb-4">
              Ready to Join Our Success Stories?
            </h3>
            <p className="text-gray-600 mb-6">
              Start your journey to medical exam success today
            </p>
            <button className="group inline-flex items-center px-8 py-4 bg-gradient-to-r from-blue-700 to-cyan-700 text-white font-semibold rounded-full transform transition-all duration-300 hover:scale-105 hover:shadow-2xl">
              Join Our Community
              <svg className="ml-2 w-5 h-5 transform group-hover:translate-x-1 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7l5 5m0 0l-5 5m5-5H6" />
              </svg>
            </button>
          </div>
        </div>
      </div>

      {/* Custom Animations */}
      <style jsx>{`
        @keyframes float {
          0%, 100% {
            transform: translateY(0px) rotate(0deg);
          }
          33% {
            transform: translateY(-15px) rotate(5deg);
          }
          66% {
            transform: translateY(5px) rotate(-5deg);
          }
        }
        
        .animate-float {
          animation: float ease-in-out infinite;
        }
      `}</style>
    </section>
  );
}

export default Testimonials;