<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class HuggingFaceService
{
    protected string $baseUrl;
    protected string $token;

    public function __construct()
    {
        // Use the chat completions endpoint
        $this->baseUrl = 'https://router.huggingface.co/v1';
        $this->token   = config('services.huggingface.token');
    }

    /**
     * Query Hugging Face using OpenAI-compatible Chat Completions API
     */
    public function query(string $model, string $prompt, array $parameters = []): array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->token,
                'Content-Type'  => 'application/json',
            ])
            ->timeout(120)
            ->post($this->baseUrl . '/chat/completions', [
                'model' => $model,
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $prompt,
                    ],
                ],
                'max_tokens' => $parameters['max_new_tokens'] ?? 1200,
                'temperature' => $parameters['temperature'] ?? 0.4,
                'top_p' => $parameters['top_p'] ?? 0.95,
            ]);

            if (!$response->successful()) {
                Log::error('HuggingFace API HTTP error', [
                    'status' => $response->status(),
                    'body'   => $response->body(),
                ]);

                return [
                    'error' => $response->json()['error'] ?? 'HuggingFace request failed',
                ];
            }

            $data = $response->json();
            
            // Transform to match your existing code expectations
            return [
                [
                    'generated_text' => $data['choices'][0]['message']['content'] ?? '',
                ]
            ];

        } catch (\Throwable $e) {
            Log::error('HuggingFace API Exception', [
                'message' => $e->getMessage(),
            ]);

            return [
                'error' => $e->getMessage(),
            ];
        }
    }
}