<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Smalot\PdfParser\Parser;
use Auth;
use App\Models\PersonalQuestion; 
use App\Models\Course;

class QuestionGeneratorController extends Controller
{
    public function ai_generateQuestions(Request $request)
    {
      
        $request->validate([
            'file' => 'required|file|mimes:pdf|max:' . env('MAX_FILE_SIZE', 51200),
            'question_count' => 'required|integer|min:1|max:50',
            'course_name' => 'required|string|max:255',
            'subject_id' => 'nullable|integer|exists:subjects,id',
            'subtopic_id' => 'nullable|integer|exists:sub_topics,id',
        ]);
        
        $courseName = $request->input('course_name');
       
        // $courseId = $course->id;
        try {
            // Store the uploaded file temporarily
            $file = $request->file('file');
            $path = $file->store('temp');
            $fullPath = Storage::path($path);

            try {
                // Extract text from PDF
                $parser = new Parser();
                $pdf = $parser->parseFile($fullPath);
                $extractedText = $pdf->getText();
                Log::info('PDF parsing successful. Text length: ' . strlen($extractedText));
            } catch (\Exception $e) {
                Log::error('PDF text extraction failed: ' . $e->getMessage());
                throw new \Exception('Failed to extract text from PDF. Please ensure the PDF contains readable text and is not password protected.');
            } finally {
                Storage::delete($path);
            }

            // Clean and validate extracted text
            $extractedText = trim($extractedText);
            if (empty($extractedText)) {
                return response()->json([
                    'error' => [
                        'message' => 'Could not extract text from the file or the file appears to be empty.',
                        'code' => 'EMPTY_TEXT',
                    ]
                ], 422);
            }

            // Validate text length for question count
            $questionCount = (int) $request->input('question_count');
            $minTextLength = $questionCount * 100; // ~100 characters per question
            if (strlen($extractedText) < $minTextLength) {
                Log::warning("Extracted text too short for $questionCount questions: " . strlen($extractedText) . " characters");
                return response()->json([
                    'error' => [
                        'message' => "The extracted text is too short to generate $questionCount questions. Please provide a PDF with more content.",
                        'code' => 'INSUFFICIENT_CONTENT',
                    ]
                ], 422);
            }

            // Split text into chunks
            $chunkSize = env('GEMINI_TEXT_CHUNK_SIZE', 300000);
            $chunks = $this->splitText($extractedText, $chunkSize);
            $questionsPerChunk = ceil($questionCount / max(1, count($chunks)));
            $allQuestions = [];
            $maxRetries = 3;
            $retryCount = 0;

            while (count($allQuestions) < $questionCount && $retryCount < $maxRetries) {
                foreach ($chunks as $index => $chunk) {
                    $chunkQuestionCount = min($questionsPerChunk, $questionCount - count($allQuestions));
                    if ($chunkQuestionCount <= 0) {
                        break;
                    }

                    // Sanitize chunk to prevent prompt injection
                    $chunk = str_replace(['"', "\n", "\r"], ['\"', ' ', ' '], $chunk);

                    // Prepare prompt
                    $prompt = "You are a question generation assistant. Generate exactly $chunkQuestionCount multiple-choice questions based on the provided content.

CRITICAL INSTRUCTIONS:
1. Return ONLY a valid JSON array - no other text, explanations, or formatting
2. Do not wrap the JSON in markdown code blocks
3. Do not include any text before or after the JSON array
4. Each question must have exactly 4 options
5. correct_answer must be an integer (0, 1, 2, or 3)

Required JSON format:
[{\"question\":\"Question text?\",\"options\":[\"A\",\"B\",\"C\",\"D\"],\"correct_answer\":0,\"explanation\":\"Why this is correct\"}]

Content:
$chunk";

                    // Call Gemini API
                    $questions = $this->callGeminiApi($prompt);
                    Log::info("Generated " . count($questions) . " questions for chunk $index in retry $retryCount");
                    $allQuestions = array_merge($allQuestions, $questions);
                    if (count($allQuestions) >= $questionCount) {
                        break;
                    }
                }
                $retryCount++;
            }

            // Trim to requested number of questions
            $allQuestions = array_slice($allQuestions, 0, $questionCount);
            Log::info("Requested $questionCount questions, generated " . count($allQuestions));
            if (count($allQuestions) < $questionCount) {
                Log::warning("Generated fewer questions than requested: " . count($allQuestions) . " of $questionCount");
            }

            // Save questions to database
            $course = Course::create([
                'name' => $courseName,
                'user_id' => Auth::id(),
                'subject_id' => $request->subject_id,
                'subtopic_id' => $request->subtopic_id
            ]);

            $savedQuestions = $this->saveQuestionsToDatabase($allQuestions, $course->id, $request->subject_id, $request->subtopic_id);
            Log::info("Saved " . count($savedQuestions) . " questions to database");
        
            return response()->json([
                'questions' => $savedQuestions,
                'message' => 'Questions generated and saved successfully'
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error generating questions: ' . $e->getMessage());
            return response()->json([
                'error' => [
                    'message' => $e->getMessage(),
                    'code' => $e->getCode() ?: 'GENERAL_ERROR',
                ]
            ], $e instanceof \Illuminate\Validation\ValidationException ? 422 : 500);
        }
    }

    /**
     * Save generated questions to the PersonalQuestion model
     */
    private function saveQuestionsToDatabase($questions, $courseId, $subjectId, $subtopicId)
    {
        $savedQuestions = [];
        $userId = Auth::id();
    
        foreach ($questions as $questionData) {
            try {
                $personalQuestion = PersonalQuestion::create([
                    'course_id' => $courseId, 
                    'user_id' => $userId,
                    'subject_id' => $subjectId,
                    'subtopic_id' => $subtopicId,
                    'question' => $questionData['question'],
                    'answer' => $questionData['correct_answer'], // index of correct answer
                    'explanation' => $questionData['explanation'],
                    'options' => json_encode($questionData['options']), // store options as JSON
                ]);
    
                // Add the ID to the question data for response
                $questionData['id'] = $personalQuestion->id;
                $savedQuestions[] = $questionData;
    
                Log::info("Saved question with ID: " . $personalQuestion->id);
            } catch (\Exception $e) {
                Log::error("Failed to save question: " . $e->getMessage() . " | Question: " . json_encode($questionData));
                // Continue with other questions even if one fails
            }
        }
    
        return $savedQuestions;
    }

    private function callGeminiApi($prompt)
    {
        $client = new Client();
        $apiKey = env('GEMINI_API_KEY');
        $apiEndpoint = env('GEMINI_API_ENDPOINT', 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent');
                if (!$apiKey) {
            throw new \Exception('Gemini API key is not configured in environment variables');
        }

        try {
            Log::info('Calling Gemini API...');
            $response = $client->post($apiEndpoint, [
                'headers' => [
                    'x-goog-api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'contents' => [
                        [
                            'parts' => [
                                ['text' => $prompt],
                            ],
                        ],
                    ],
                    'generationConfig' => [
                        'temperature' => 0.3,
                        'maxOutputTokens' => env('GEMINI_MAX_OUTPUT_TOKENS', 8192),
                        'responseMimeType' => 'application/json',
                        'responseSchema' => [
                            'type' => 'array',
                            'items' => [
                                'type' => 'object',
                                'properties' => [
                                    'question' => ['type' => 'string'],
                                    'options' => [
                                        'type' => 'array',
                                        'items' => ['type' => 'string'],
                                        'minItems' => 4,
                                        'maxItems' => 4,
                                    ],
                                    'correct_answer' => ['type' => 'integer', 'minimum' => 0, 'maximum' => 3],
                                    'explanation' => ['type' => 'string'],
                                ],
                                'required' => ['question', 'options', 'correct_answer', 'explanation'],
                            ],
                        ],
                    ],
                ],
                'timeout' => 60,
            ]);

            $result = json_decode($response->getBody(), true);
            $content = $result['candidates'][0]['content']['parts'][0]['text'] ?? null;

            if (!$content) {
                Log::error('No content returned from Gemini API');
                throw new \Exception('No content returned from AI service');
            }

            Log::info('Gemini API raw response: ' . substr($content, 0, 500) . '...');

            // Clean and parse JSON
            $content = trim($content);
            $questions = json_decode($content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('JSON parsing error: ' . json_last_error_msg() . ' | Content: ' . $content);
                throw new \Exception('Could not parse AI response. The AI may have returned malformed JSON.');
            }

            if (!is_array($questions)) {
                Log::error('Response is not an array: ' . gettype($questions));
                throw new \Exception('AI response is not an array of questions');
            }

            // Validate question structure
            foreach ($questions as $index => $question) {
                if (!isset($question['question'], $question['options'], $question['correct_answer'], $question['explanation'])) {
                    Log::error("Question at index $index is missing required fields: " . json_encode($question));
                    throw new \Exception("Question at index $index is missing required fields");
                }
                if (!is_array($question['options']) || count($question['options']) !== 4) {
                    Log::error("Question at index $index does not have exactly 4 options: " . json_encode($question['options']));
                    throw new \Exception("Question at index $index does not have exactly 4 options");
                }
                if (!is_int($question['correct_answer']) || $question['correct_answer'] < 0 || $question['correct_answer'] > 3) {
                    Log::error("Question at index $index has invalid correct_answer value: " . $question['correct_answer']);
                    throw new \Exception("Question at index $index has invalid correct_answer value");
                }
            }

            Log::info('Successfully parsed ' . count($questions) . ' questions');
            return $questions;

        } catch (\GuzzleHttp\Exception\RequestException $e) {
            Log::error('Gemini API request failed: ' . $e->getMessage());
            if ($e->hasResponse()) {
                Log::error('API Error Response: ' . $e->getResponse()->getBody()->getContents());
            }
            throw new \Exception('Failed to communicate with AI service. Please try again later.');
        } catch (\GuzzleHttp\Exception\ConnectException $e) {
            Log::error('Connection to Gemini API failed: ' . $e->getMessage());
            throw new \Exception('Cannot connect to AI service. Please check your internet connection.');
        }
    }

    private function truncateText($text, $maxLength = 1000000)
    {
        if (strlen($text) <= $maxLength) {
            return $text;
        }
        $truncated = substr($text, 0, $maxLength);
        $lastPeriod = strrpos($truncated, '.');
        $lastNewline = strrpos($truncated, "\n");
        $cutoff = max($lastPeriod, $lastNewline);
        if ($cutoff !== false) {
            $truncated = substr($truncated, 0, $cutoff + 1);
        } else {
            $truncated .= '...';
        }
        Log::info('Text truncated to ' . strlen($truncated) . ' characters at logical boundary');
        return $truncated;
    }

    private function splitText($text, $chunkSize = 300000)
    {
        $chunks = [];
        while (strlen($text) > 0) {
            $chunk = $this->truncateText($text, $chunkSize);
            $chunks[] = $chunk;
            $text = substr($text, strlen($chunk));
        }
        Log::info('Text split into ' . count($chunks) . ' chunks');
        return $chunks;
    }
    public function user_questions(Request $request){
        $data = PersonalQuestion::all();
        return $data;
    }
}