<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use App\Models\Subscription;
use Carbon\Carbon;
use Auth;
use App\Models\User;

class PaymentController extends Controller
{
    public function verifyPayment(Request $request)
    {
      
        $request->validate([
            'reference' => 'required|string',
            'plan' => 'required|string',
            'email' => 'required|email',
        ]);

        // Call Paystack verify API
        $response = Http::withToken(env('PAYSTACK_SECRET_KEY'))
            ->get("https://api.paystack.co/transaction/verify/{$request->reference}");

        if (!$response->successful()) {
            return response()->json(['message' => 'Paystack verification failed'], 500);
        }

        $data = $response->json();

        if ($data['status'] !== true || $data['data']['status'] !== 'success') {
            return response()->json(['message' => 'Transaction not successful'], 400);
        }

        // Plan durations in months
        $durations = [
            '1_month' => 1,
            '3_months' => 3,
            '6_months' => 6,
        ];

        $duration = $durations[$request->plan] ?? 1;

        // Save subscription
        $subscription = Subscription::create([
            'user_id' => Auth::id(),
            'plan' => $request->plan,
            'amount' => $data['data']['amount'] / 100,
            'reference' => $request->reference,
            'status' => 'active',
            'starts_at' => Carbon::now(),
            'ends_at' => Carbon::now()->addMonths($duration),
        ]);
        $user = User::findOrFail(Auth::id());
        $user->is_premium = 1;
        $user->save();

        return response()->json([
            'message' => 'Subscription verified and saved successfully',
            'subscription' => $subscription,
        ]);
    }
    public function getSubscriptionStatus(Request $request)
    {
        $user = Auth::user();
        
        // Get the user's active subscription
        $subscription = Subscription::where('user_id', $user->id)
            ->where('status', 'active')
            ->where('ends_at', '>', Carbon::now())
            ->latest()
            ->first();

        if ($subscription) {
            return response()->json([
                'has_subscription' => true,
                'status' => $subscription->status,
                'plan' => $subscription->plan,
                'starts_at' => $subscription->starts_at,
                'ends_at' => $subscription->ends_at,
                'amount' => $subscription->amount,
            ]);
        }

        return response()->json([
            'has_subscription' => false,
            'status' => null,
            'plan' => null,
        ]);
    }
    public function admin_premium(Request $request)
{
    // Validate the request
    $validated = $request->validate([
        
        'duration' => 'required|integer|min:1|max:12'
    ]);

    $userId = $request->id;
    $duration = $validated['duration'];
    
    // Calculate amount based on duration
    $amounts = [
        1 => 1500,
        3 => 4000,
        6 => 7500,
        12 => 14000
    ];
    
    $amount = $amounts[$duration] ?? ($duration * 1500);
    
    // Check if user already has an active subscription
    $existingSubscription = Subscription::where('user_id', $userId)
        ->where('ends_at', '>', Carbon::now())
        ->first();
    
    if ($existingSubscription) {
        // Extend existing subscription
        $existingSubscription->ends_at = Carbon::parse($existingSubscription->ends_at)->addMonths($duration);
        $existingSubscription->save();
        
        $subscription = $existingSubscription;
    } else {
        // Create new subscription
        $subscription = Subscription::create([
            'user_id' => $userId,
            'plan' => $duration . '_month' . ($duration > 1 ? 's' : ''),
            'amount' => $amount,
            'reference' => 'ADMIN_PREMIUM_' . strtoupper(uniqid()),
            'status' => 'active',
            'starts_at' => Carbon::now(),
            'ends_at' => Carbon::now()->addMonths($duration),
        ]);
    }
    
    // Update user's premium status
    $user = User::findOrFail($userId);
    $user->is_premium = true;
    $user->save();
    
    return response()->json([
        'message' => 'Premium access granted successfully',
        'subscription' => $subscription,
        'user' => $user
    ], 200);
}


public function remove_premium($userId)
{
    $user = User::findOrFail($userId);
    
    // Cancel all active subscriptions
    Subscription::where('user_id', $userId)
        ->where('ends_at', '>', Carbon::now())
        ->update([
            'status' => 'cancelled',
            'ends_at' => Carbon::now()
        ]);
    
    // Update user status
    $user->is_premium = false;
    $user->save();
    
    return response()->json([
        'message' => 'Premium access removed successfully',
        'user' => $user
    ], 200);
}
}
