<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Course;
use App\Models\PersonalQuestion;
use Auth;
class CourseController extends Controller
{
    //
    public function get_course(Request $request)
    {
        try {
            $userId = Auth::id();
            $perPage = $request->get('per_page', 12);
    
            $query = Course::where('user_id', $userId)
                ->with(['subject', 'subtopic'])
                ->withCount('personalQuestions')->orderBy('created_at', 'desc');
    
            if ($request->filled('subject_id')) {
                $query->where('subject_id', $request->subject_id);
            }
    
            if ($request->filled('subtopic_id')) {
                $query->where('subtopic_id', $request->subtopic_id);
            }
    
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhereHas('subject', fn($sq) => $sq->where('name', 'like', "%{$search}%"))
                      ->orWhereHas('subtopic', fn($sq) => $sq->where('name', 'like', "%{$search}%"));
                });
            }
    
            $paginated = $query->paginate($perPage);
    
            $formatted = $paginated->getCollection()->map(function ($course) {
                return [
                    'id'                       => $course->id,
                    'name'                     => $course->name,
                    'subject'                  => $course->subject?->name ?? null,
                    'subject_id'               => $course->subject_id,
                    'subtopic'                 => $course->subtopic?->name ?? null,
                    'subtopic_id'              => $course->subtopic_id,
                    'personal_questions_count' => $course->personal_questions_count,
                    'created_at'               => $course->created_at,
                ];
            });
    
            return response()->json([
                'success'      => true,
                'courses'      => $formatted,
                'current_page' => $paginated->currentPage(),
                'last_page'    => $paginated->lastPage(),
                'total'        => $paginated->total(),
                'per_page'     => $paginated->perPage(),
            ], 200);
    
        } catch (\Exception $e) {
            \Log::error('get_course error: ' . $e->getMessage(), ['exception' => $e]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch courses.',
            ], 500);
        }
    }
    public function get_public_quiz(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 12);
            $query = Course::with(['subject', 'subtopic'])
                ->withCount('personalQuestions')
                ->where('status', 1)
                ->orderBy('created_at', 'desc');
    
            if ($request->filled('subject_id')) {
                $query->where('subject_id', $request->subject_id);
            }
    
            if ($request->filled('subtopic_id')) {
                $query->where('subtopic_id', $request->subtopic_id);
            }
    
            if ($request->filled('search')) {
                $search = $request->search;
    
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhereHas('subject', function ($sq) use ($search) {
                          $sq->where('name', 'like', "%{$search}%");
                      })
                      ->orWhereHas('subtopic', function ($sq) use ($search) {
                          $sq->where('name', 'like', "%{$search}%");
                      });
                });
            }
    
            // ✅ Paginate LAST
            $paginated = $query->paginate($perPage);
    
            // ✅ Format response
            $formatted = $paginated->getCollection()->map(function ($course) {
                return [
                    'id'                       => $course->id,
                    'name'                     => $course->name,
                    'subject'                  => $course->subject?->name,
                    'subject_id'               => $course->subject_id,
                    'subtopic'                 => $course->subtopic?->name,
                    'subtopic_id'              => $course->subtopic_id,
                    'personal_questions_count' => $course->personal_questions_count,
                    'created_at'               => $course->created_at,
                ];
            });
    
            return response()->json([
                'success'      => true,
                'courses'      => $formatted,
                'current_page' => $paginated->currentPage(),
                'last_page'    => $paginated->lastPage(),
                'total'        => $paginated->total(),
                'per_page'     => $paginated->perPage(),
            ], 200);
    
        } catch (\Exception $e) {
            \Log::error('get_public_quiz error: ' . $e->getMessage(), ['exception' => $e]);
    
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch courses.',
            ], 500);
        }
    }
    
    public function add_course(Request $request){
        $data = $request->all();
        try {
            $course = Course::create([
                'name' => $data['name'],
                
                'user_id' => Auth::id(),
            ]);

            return response()->json($course, 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to add course: ' . $e->getMessage(),
            ], 500);
        }
        
    }
    public function get_personal_questions(Request $request)
    {
        try {
            // Validate the request
            $request->validate([
                'course_id' => 'required|integer|exists:courses,id'
            ]);
    
            $courseId = $request->course_id;
    
            // Fetch questions with proper error handling
            $questions = PersonalQuestion::where('course_id', $courseId)
                ->select([
                    'id',
                    'user_id',
                    'course_id',
                    'question',
                    'answer',
                    'explanation',
                    'options',
                    'created_at',
                    'updated_at'
                ])
                ->orderBy('created_at', 'asc')
                ->get();
    
            // Check if questions exist
            if ($questions->isEmpty()) {
                return response()->json([], 200);
            }
    
            // Transform questions to match frontend expectations
            $transformedQuestions = $questions->map(function ($question) {
                // Parse the JSON options string
                $optionsArray = json_decode($question->options, true);
                
                // Handle case where JSON parsing fails or returns null
                if (!is_array($optionsArray)) {
                    \Log::warning('Failed to parse options JSON for question ID: ' . $question->id, [
                        'options_raw' => $question->options
                    ]);
                    $optionsArray = [];
                }
    
                // Convert numeric answer to letter (0=A, 1=B, 2=C, 3=D)
                $correctAnswerText = '';
                if (is_numeric($question->answer)) {
                    $answerIndex = (int)$question->answer;
                    $correctAnswerText = $optionsArray[$answerIndex] ?? '';
                }
    
                return [
                    'id' => $question->id,
                    'user_id' => $question->user_id,
                    'course_id' => (int)$question->course_id,
                    'question' => $question->question,
                    'options' => $optionsArray, // Send as array for frontend
                    'answer' => $correctAnswerText, // Send the actual answer text
                    'numeric_answer' => (int)$question->answer, // Keep original for reference
                    'explanation' => $question->explanation,
                    'created_at' => $question->created_at,
                    'updated_at' => $question->updated_at,
                ];
            });
    
            return response()->json($transformedQuestions, 200);
    
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
    
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Error fetching personal questions: ' . $e->getMessage(), [
                'course_id' => $request->course_id ?? null,
                'user_id' => auth()->id() ?? null,
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'message' => 'No questions found for this course'
            ], 404);
        }
    }
public function delete_course(Request $request){
    $id = $request->id;
    try {
       $course = Course::findOrFail($id);
       $course->delete();
       PersonalQuestion::where('course_id', $id)->delete();
       return response()->json([
        'message' => "Course and related questions deleted successfully",
        'course_id' => $id
       ], 200);

    
    } catch (\Exception $e) {
        return response()->json([
            'message' => 'Failed to delete course',
            'error' => $e->getMessage()
        ], 500);
    }
}
public function edit_course(Request $request){
    try {
        $id = $request->id;
        Course::where('id', $id)->update(['name' => $request->name]);
        return response()->json([
            'message' => 'course Edited successfully',

        ], 200);
    } catch (\Exception $e) {
       return response()->json([
        'message' => 'sorry an error occured',
        'error', $e->getMessage()
       ], 500);
    }
}
    public function get_personal_questions_numeric(Request $request)
    {
        try {
            $request->validate([
                'course_id' => 'required|integer|exists:courses,id'
            ]);
    
            $questions = PersonalQuestion::where('course_id', $request->course_id)
                ->orderBy('created_at', 'asc')
                ->get();
    
            if ($questions->isEmpty()) {
                return response()->json([], 200);
            }
    
            $transformedQuestions = $questions->map(function ($question) {
                $optionsArray = json_decode($question->options, true) ?? [];
    
                return [
                    'id' => $question->id,
                    'user_id' => $question->user_id,
                    'course_id' => (int)$question->course_id,
                    'question' => $question->question,
                    'options' => [
                        'A' => $optionsArray[0] ?? '',
                        'B' => $optionsArray[1] ?? '',
                        'C' => $optionsArray[2] ?? '',
                        'D' => $optionsArray[3] ?? '',
                    ],
                    'answer' => (int)$question->answer, // Keep as numeric (0,1,2,3)
                    'explanation' => $question->explanation,
                    'created_at' => $question->created_at,
                    'updated_at' => $question->updated_at,
                ];
            });
    
            return response()->json($transformedQuestions, 200);
    
        } catch (\Exception $e) {
            \Log::error('Error fetching personal questions: ' . $e->getMessage());
            return response()->json([
                'message' => 'No questions found for this course'
            ], 404);
        }
    }
}